;; The Paranoid Imaginarium of Roger L'Estrange
;; Michael Gavin, University of South Carolina
;;

;; Breeds created at initialization.
breed [readers reader]
breed [booksellers bookseller]
breed [books book]
breed [manuscripts manuscript]
breed [informers informer]

globals [ 
  bookshops-raided           ; record of informer behavior
  authors-executed           ; record of informer behavior
  crowds-dispersed           ; record of informer behavior
  book-age-threshold         ; global variable that determines book life-span
  tolerated-publications     ; global variable that determines informer behavior
  ]

turtles-own [ opinion ]      ; all turtles (books and people) have opinions on scale of -1 , 1

booksellers-own [
  my-manuscripts             ; (agentset) sellers keep track of owned manuscripts
  my-books                   ; (agentset) maintain inventory
  imprisoned                 ; (true/false) can be imprisoned
  time-served                ; (numerical) keep track of time served
  ]

readers-own [
  my-books                   ; (agentset) readers carry books with them
  my-manuscripts             ; (agentset) readers sometimes write new manuscripts
  publications               ; (numerical) number of successful publications
  initial-expressivity       ; (numerical) baseline desire to write, normal distribution
  expressivity               ; (numerical) current fluctuating state of desire to write
  defensiveness              ; (numerical) determines how readers respond to others' opinions
  talent                     ; (numerical) determines quality of a reader's manuscripts
  initial-money              ; (numerical) baseline money level, normal distribution
  money                      ; (numerical) current fluctuating state of money
  ]

manuscripts-own [
  my-owner                   ; (agentset) current owner of manuscript
  my-author                  ; (agentset) creator of manuscript
  my-copies                  ; (agentset) books hatched from manuscript 
  book-age                   ; (numerical) number of ticks since hatch
  title                      ; (numerical) randomly generated ID number
  editions                   ; (numerical) record of printings
  appeal                     ; (numerical) appeal of text, normal distribution
  profit                     ; (numerical) record of profit generated for seller   
  price                      ; (numerical) price, normal distribution
  ]


books-own [
  my-owner                   ; (agentset) current owner
  my-author                  ; (agentset) original creator
  book-age                   ; (numerical) number of ticks since hatch
  appeal                     ; (numerical) appeal of text, taken from parent manuscript
  price                      ; (numerical) price of text, taken from parent manuscript
  title                      ; (numerical) title of text, taken from parent manuscript
  ]

to setup
  ca
  
  set bookshops-raided 0
  set authors-executed 0
  set crowds-dispersed 0
  set book-age-threshold 50         ; global variable (could be converted to slider bar)
  set tolerated-publications 2      ; global variable (could be converted to slider bar)
  
  ; creates readers (randomized properties)
  set-default-shape readers "person"
  create-readers number-of-readers [
    set opinion 1 - random-float 2
    ifelse opinion < -.9 
       [ set color red ] 
       [ set color violet + opinion ] 
    set talent random 10
    set initial-expressivity (1 - random-float 1)
    set expressivity initial-expressivity
    let defensiveness-randomization random-float 10
    ifelse defensiveness-randomization > 5
      [ set defensiveness one-of [ 1 -1 ] ]
      [ set defensiveness 1]
    set initial-money random 10
    set money initial-money
    set my-books nobody
    set my-manuscripts nobody
    set publications 0
    set size 2
    setxy random-xcor random-ycor
  ]
  
  ; creates readers with pre-determined properties typical of "radicals"
  create-readers number-of-radicals [
    set opinion -1                          ; strong negative opinion
    set talent 9                            ; high talent
    set initial-expressivity .95            ; very expressive
    set expressivity initial-expressivity
    set defensiveness -1                    ; very defensive
    set initial-money 0
    set money random 10
    set my-books nobody
    set my-manuscripts nobody
    set publications 0
    set size 2
    setxy random-xcor random-ycor
  ]
 
  ; creates readers with pre-determined properties typical of "propagandists" 
    create-readers number-of-propagandists [
    set opinion 1                            ; strong negative opinion
    set talent 9                             ; high talent
    set initial-expressivity .95             ; very expressive
    set expressivity initial-expressivity
    set defensiveness -1                     ; very defensive
    set initial-money 0
    set money random 10
    set my-books nobody
    set my-manuscripts nobody
    set publications 0
    set size 2
    setxy random-xcor random-ycor
  ]
  
  ; create informers
  set-default-shape informers "person"
  create-informers number-of-informers [ 
    set color white
    set size 2
    set opinion 0
    setxy random-xcor random-ycor ]
  
  ; create booksellers
  set-default-shape booksellers "house"
  create-booksellers number-of-booksellers [
    set color brown
    set size 3
    set my-manuscripts nobody
    set imprisoned false
    setxy random-xcor random-ycor
    ]
  
  ; create books
  set-default-shape books "box"
  create-books number-of-readers [ 
    set opinion 1 - random-float 2
    ifelse opinion < -.9 [ set color red ] [ set color violet + opinion ] 
    set my-owner one-of booksellers
    setxy ([xcor] of my-owner) ([ycor] of my-owner)
  ]
  ; set shapes for manuscripts (none created at initialization)
  set-default-shape manuscripts "star"

  reset-ticks
end

; Go activates readers and booksellers.
to go
  ask readers [ 
    manage-books
    update-color
    discuss
    move
    buy-books ]
  ask booksellers [ check-for-booksellers-freedom ]
  ask informers [ determine-behavior ]
  if percentage-red >= 50 [stop]
  if not any? readers with [ opinion < -0.5 and defensiveness > 0 ] [ stop ]
  if ticks = 3000 [stop]
  tick
end

; informer
to determine-behavior
  let potential-rebels readers with [color = red] in-radius informer-radius
  let seditious-books books with [color = red] in-radius informer-radius
  if count potential-rebels + count seditious-books > 1 [ hunt-for-books ]
end

; Hunting for books means that informers move around the screen and look for red booksellers
to hunt-for-books
  rt random 20
  lt random 20
  fd .5
  let booksellers-to-oppress booksellers with [color = red] in-radius informer-radius
  if any? booksellers-to-oppress [ 
      ask booksellers-to-oppress [ 
        ask my-books [ die ]
        ask my-manuscripts [ die ]
        go-to-jail
        set bookshops-raided bookshops-raided + 1 ]
        ]
   if count ( readers with [color = red ] in-radius informer-radius ) > 2 [ 
     ask readers with [color = red ] in-radius informer-radius [ setxy random-xcor random-ycor ]
     set crowds-dispersed crowds-dispersed + 1
  ]
  if hunt-authors = true [
  let authors-to-oppress readers with [ color = red and publications > tolerated-publications ]
  if any? authors-to-oppress in-radius informer-radius [ ask one-of authors-to-oppress [ 
        hatch 1 [ set opinion 1 - random-float 2]
          ask my-books [die]
          ask my-manuscripts [die]
          die ] 
        set authors-executed authors-executed + 1]
  ]
  end
  
; booksellers do this when raided
to go-to-jail
  set imprisoned true
  set color black
  set time-served 0
  serve-time
end

; while imprisoned, booksellers count down their sentences
to serve-time
  set time-served time-served + 1
  if time-served >= sentence-length [
    set imprisoned false ]
end

; if they're free, sellers manage their inventory
to check-for-booksellers-freedom
  ifelse imprisoned = false [ manage-inventory ] [ serve-time ]
end

; Booksellers manage their inventory by looking through the agentset
; of their books and manuscripts. They ask their books to "age" and
; they look for opportunities to reprint existing manuscripts or
; publish new ones.
to manage-inventory
  set my-books books with [my-owner = myself]
  ifelse any? my-books with [color = red] [set color red] [set color brown]
  ask my-books [
    setxy ( [xcor] of myself) ( [ycor] of myself)
    age]
  set my-manuscripts manuscripts with [my-owner = myself]
  ask my-manuscripts [ setxy ( [xcor] of myself) ( [ycor] of myself) ]
  let old-manuscripts my-manuscripts with [editions > 5]
  ask old-manuscripts [die]
  if count my-manuscripts > 0 and count my-books < (2 * 25) [ reprint ]
end
  
; When first published, books are given an "appeal" score. Every tick,
; that score depreciates according to an adjustable rate. If the books
; lose all consumer appeal, the bookseller will sell them off at a 
; cheap rate as paper-capital.
to age
  if my-owner = one-of booksellers [
  set book-age book-age + 1
  set appeal appeal - depreciation-rate
  if book-age > book-age-threshold [ 
    let my-original manuscripts with [title = [title] of myself]
       ask my-original [ die ]
    die
    ] ]
  if my-owner = one-of readers [
    set book-age book-age + 1
    if book-age > book-age-threshold [die ] ]
end
  
; "Reprint" is a command that booksellers give to their 
; manuscripts. Each manuscript looks through the agentset of books that
; share their title. If the capital of books has been depleted, and if the
; title has been profitable, it will be reprinted. If too many copies of the
; book remain unsold when the title loses appeal, all copies of the book are
; pulped, and the manuscript is retired. -- NOTE: As written, this command
; leads to overproduction and certain bankruptcy for booksellers. It must be improved.
to reprint
  if count my-manuscripts >= 1 
    [ ask max-one-of my-manuscripts [my-copies] [ 
        set editions editions + 1 
        hatch-books 25 ] ]
end

; Readers move about randomly, always updating their "my-books" agentset and
; carrying those books along with them.
to manage-books
  set my-books books with [my-owner = myself]
  ask my-books [ setxy ([xcor] of myself) ([ycor] of myself) ]
  if any? my-books [read-books
    ask my-books [                                               ;
    set appeal appeal - (depreciation-rate / 2)                ; but their holdings gradually become obsolete and die.
    if appeal <= 0 [die] ]
    ask my-manuscripts [
    set appeal appeal - (depreciation-rate / 5)
    if appeal <= 0 [die] ]
  ]
  
  set my-manuscripts manuscripts with [my-owner = myself]
  ask my-manuscripts [ setxy ([xcor] of myself) ([ycor] of myself) ]
  if any? my-manuscripts [solicit-publishers]
  
  set money money + income-rate
  set expressivity expressivity * 1.001
  if expressivity >= 1 [ compose ]
end

; Here is the major function that this version includes. The readers trade
; opinions as they more around.
to update-color
  
  ifelse opinion < -.9 
    [ set color red ] 
    [ set color violet + opinion ]
end

; When agents "discuss" they update their opinion score based on the mean of all
; turtles within their radius.
to discuss
  let discussants turtles in-radius gossip-radius
  let zeitgeist mean [ opinion ] of discussants
  ifelse opinion * zeitgeist > 0 
    [ set opinion opinion + ( zeitgeist * .001 ) ]
    [ set opinion opinion + ( zeitgeist * .001 * defensiveness)  ]
  if opinion < -1 [ set opinion -1 ]
  if opinion > 1 [ set opinion 1]
end


; This is a simple rule for updating readers' opinion. Basically, they read their books
; every click. If both the "discuss" and "read-books" procedures are called, this
; basically counts the books twice. This is probably not an ideal way of dealing with
; books, and I think these two procedures should be joined into a single "update-opinion"
; function.
to read-books
  let my-reading mean [opinion] of my-books
  ifelse opinion * my-reading > 0 
    [ set opinion opinion + ( my-reading * .001 )  ]
    [ set opinion opinion + ( my-reading * .001 * defensiveness)  ]
end 

to move
  ifelse ( opinion < 0 and money > 10 )  [ 
    let books-for-me books with [ color = red and my-owner = one-of booksellers ]
    if any? books-for-me [ face min-one-of books-for-me [ distance myself ] ]
    rt random 50
    lt random 50 ]
  [ rt random 50
    lt random 50 ]
  ifelse (count readers with [color = [color] of myself] in-radius gossip-radius) > 2
    [ fd 0.5 ]
    [ fd 1.5 ]
end
; readers with manuscripts are always on the lookout for publishers
to solicit-publishers
  if any? booksellers in-radius 3 [        ; notice that publishing radius is set arbitrarily to 3. could be converted to slider bar
  let publisher one-of booksellers in-radius 3
  let offered-manuscript max-one-of my-manuscripts [appeal]
  if [appeal] of offered-manuscript > [price] of offered-manuscript [
       ask offered-manuscript [ set my-owner publisher ]
       set publications publications + 1
     ] ]
end

; when "expressivity" reaches 1, readers compose manuscripts
; note that "opinion" is not explicitly mentioned below, but that
; manuscripts automatically have the opinion score of their parent turtles     
to compose
  hatch-manuscripts 1 [
    set size 1
    set my-author myself
    set my-owner myself
    set book-age 0
    set title random-float 100
    set appeal ([talent] of my-author - random-float 10)
    set price 10 - random-float 9
    ]
  set expressivity initial-expressivity
end

; When readers buy-books, they create an agentset "books-to-buy" of
; nearby books. Eligible books must be owned by a bookseller and be priced below the
; reader's money level. Among the bookseller's collection, the reader will focus on books
; of the same color or books that are especially appealing. Incurious readers are
; not likely to purchase any book outside their preferred type (color). From the "books-to-buy" 
; agentset, the reader chooses one book at random and buys it.
   
to buy-books
  let books-to-buy books in-radius 1 with [     ; creates agentset of nearby books that meet one of the following two sets of  
    (my-owner = one-of booksellers with [ imprisoned = false ] and          ; criteria: must be owned by a bookseller, same color, and affordable
    color = [color] of myself and               ; 
    price <= [money] of myself) or              ; or, 
    (my-owner = one-of booksellers and          ; book's appeal (randomly adjusted) must overcome the reader's incuriosity.
    (appeal + random-float 2 - random-float 2) >= price and 
    price <= [money] of myself)
    ]
  
  let chosen-book one-of books-to-buy           ; Among the eligible books, reader chooses one and
  if chosen-book != nobody [
    ask chosen-book [
      set my-owner myself             ; asks chosen-book to update its "my-owner" variable
      let my-original manuscripts with [title = [title] of myself]
        ask my-original [ set my-copies my-copies + 1 ] 
       ]
    
    set money money - [price] of chosen-book    ; reader's money is depleted
    
    ask manuscripts with [title = [title] of chosen-book] [
      set profit profit + [price] of chosen-book]    ; money is subtracted, and the purchase is complete.
  ]
end

to-report percentage-red
  let R count readers with [color = red]
  let A count readers
    report  R / A * 100
end

;; 
;;
;;
;; END
;;
;;
;;
@#$#@#$#@
GRAPHICS-WINDOW
235
27
674
487
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
5
16
60
49
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
64
16
119
49
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
6
151
178
184
depreciation-rate
depreciation-rate
.005
.1
0.1
.001
1
NIL
HORIZONTAL

SLIDER
6
207
181
240
number-of-readers
number-of-readers
50
100
80
1
1
NIL
HORIZONTAL

SLIDER
9
253
157
286
number-of-booksellers
number-of-booksellers
1
10
7
1
1
NIL
HORIZONTAL

BUTTON
125
15
180
48
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
106
179
139
income-rate
income-rate
.001
.2
0.2
.001
1
NIL
HORIZONTAL

SLIDER
7
296
179
329
gossip-radius
gossip-radius
0
5
2
1
1
NIL
HORIZONTAL

PLOT
692
115
906
259
Purple = People ; Black = Books
NIL
NIL
0.0
10.0
-1.0
1.0
true
false
"" ""
PENS
"pen-2" 1.0 0 -8630108 true "" "if any? readers with [color != red] [plot mean [opinion] of readers with [color != red ]]"
"pen-1" 1.0 0 -16777216 true "" "plot mean [opinion] of books"

SLIDER
8
58
189
91
number-of-informers
number-of-informers
0
10
3
1
1
NIL
HORIZONTAL

SLIDER
5
430
177
463
informer-radius
informer-radius
0
5
3
1
1
NIL
HORIZONTAL

SLIDER
5
335
177
368
sentence-length
sentence-length
10
100
50
1
1
NIL
HORIZONTAL

SLIDER
740
20
917
53
number-of-radicals
number-of-radicals
0
10
4
1
1
NIL
HORIZONTAL

MONITOR
933
175
990
220
Raids
bookshops-raided
0
1
11

MONITOR
931
227
1018
272
Executions
authors-executed
0
1
11

MONITOR
933
282
1059
327
Crowds Dispersed
crowds-dispersed
0
1
11

SWITCH
33
384
174
417
hunt-authors
hunt-authors
1
1
-1000

SLIDER
744
70
962
103
number-of-propagandists
number-of-propagandists
0
10
0
1
1
NIL
HORIZONTAL

PLOT
696
270
896
420
Books in the World
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -8630108 true "" "plot count books with [color != red]"
"pen-1" 1.0 0 -2674135 true "" "plot count books with [color = red]"

MONITOR
927
116
992
161
% Red
percentage-red
0
1
11

@#$#@#$#@
# The Paranoid Imaginarium of Roger L'Estrange

## WHAT IS IT?

This model simulates the censorship practices of the seventeenth-century book trade in London under state licenser Roger L'Estrange. It creates a field of bookshops and highly politicized readers. It is designed to simulate, not so much the history of the seventeenth-century print marketplace, but the history of how censors imagined the reading public they were tasked with policing.

The model is designed to simulate the impact that seditious publication might have had on a reading public during the hand-press era. As politically engaged books are circulated through a population, that population becomes either more or less radical, perhaps leading towards the revolution that censors fear and hope to prevent.

## HOW TO USE IT

The SETUP button initializes the model. Brown bookshops are distributed around the world, as are a number of readers and informers. Slider bars control the number of each breed created at initialization. Additionally, the slider bars on the top right, "number-of-radicals" and "number-of-propagandists" create additional "reader" turtles with special properties.

Readers' behavior is determined by several variables: color, expressivity, and money. All readers will seek out books of their own color. They gain money every tick and use money to buy books. Each reader also has an "expressivity" score that increases gradually with each tick; when the expressivity score reaches 1, the reader produces a manuscript and its expressivity score is returned to baseline level. Readers are defined primarily by their "opinion" score. As they move about, the talk to each other, read books, and their opinions gradually change. When a reader's opinion score falls below -.9, it is considered a "radical."

Manuscripts are composed by readers and given to booksellers. They are "published" by hatching a number of books that share their variables. They maintain a "profit" score, which increases whenever a copy is sold, and profitable manuscripts are chosen for reprinting.

Booksellers collect manuscripts and publish them as books. Books are passive objects owned by booksellers and readers. They have appeal and a price. Books slowly lose their appeal, and if they remain too long in the booksellers' inventory, they die. They are hatched in identical batches from manuscripts. Like readers, they have an opinion score and if they are radical (-.9 or lower) they are marked by the color red.

At the start of each tick, informers check to see if there are radical readers or books in their vicinity. If so, they move around and hunt for seditious books. If informers find a red colored bookstore in their region, that destroy all the books and manuscripts and send the bookseller to prison. If "hunt-authors" is activated, informers will also kill radicals that have published multiple books.

The simulation will STOP if one of three conditions are met. If more than 50% of the population turns radical (revolution succeeds), if all persuadable readers are rendered docile (revolution fails), or if the system reaches 3,000 ticks (equilibrium/stability).

## THINGS TO NOTICE

As the model runs, watch to see if the revolution succeeds. What factors seemed to influence the outcome?

How does the system react when a radical book is published? 

Follow the informers. 

## THINGS TO TRY

Try varying the number of "radicals" or "propagandists" at initialization. How do they change behavior of the system?

Try toggling the "hunt authors" option. How does turning on the "hunt authors" option affect the outcome?

Try increasing or decreasing the income rate. How does that affect reader behavior?

## EXTENDING THE MODEL

In the seventeenth century, informers hunted authors much the way law enforcement today tracks drug dealers--they captured peddlers on the street and then tried to get information up the chain of supply. In this model, there is no anonymity, and informers are able to identify and execute radical authors. A procedure could be included, when informers raid bookstores, of gaining information from the sellers about the authors of the books they distribute.

## CREDITS AND REFERENCES

The model was inspired most directly by:
Jody Greene. <em>The Trouble with Ownership: Literary Property and Authorial Liability in England, 1660-1730</em>. Philadelphia: University of Pennsylvania Press, 2005.

## Copyright

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

This model was created by Michael Gavin, University of South Carolina. 2014.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>count books with [color != [color] of my-reader] &lt;= 20</exitCondition>
    <metric>count books with [color != [color] of my-reader]</metric>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
