extensions [
  nw]

turtles-own [
  adopted?               ; whether or not the turtles have adopted the innovation
  pers-one               ; likelihood of an adopted turtle choosing link neighbor who has not adopted and trying to persuade them
  attitude               ; attitude towards adopting the behavior
  pers-power             ; how much a turtle adds to the attitude of another turtle if they target it
  adopt-friends          ; how many link neighbors have adopted, this is for the social influence variable
  non-adopt-friends      ; how many link neighbors have not adopted
  message ]              ; keeps track of message most recently received

globals [
   status-quo?           ; sets up the counter to count change for a stop condition
   status-quo-counter
  initial-degree  ;these next four are for the variables of the seeded agents
  initial-between
  initial-pers-power
  initial-pers-likely
  adopted-at-tick
  init-adopted2

]

to setup
  ca
  reset-ticks
  set status-quo? false
  set status-quo-counter 0                                                              ; for the stop code for no change in attitude
  nw:generate-preferential-attachment turtles links num-agents min-links                ; sets up preferential attachment network
  repeat 30 [ layout-spring turtles links 0.2 5 1 ]                                     ; makes the network look nicer

  ;new code assigning adoption
  ask turtles [set adopted? false]
  set init-adopted2 ((num-agents) * (percent-adopted-start))
  ask n-of init-adopted2 turtles [set adopted? true]

  ask turtles [
  set shape "person"
  set message 0
  set-pers-one
  set-attitude
  set-pers-power
  recolor]

  ;these four calculate the values for the seeded agents
  set initial-degree mean ( [ count link-neighbors ] of turtles with [ adopted? = true ] )
  set initial-between mean ( [ nw:betweenness-centrality ] of turtles with [ adopted? = true ] )
  set initial-pers-power mean [pers-power] of turtles with [adopted? = true]
  set initial-pers-likely mean [pers-one] of turtles with [adopted? = true]
  set adopted-at-tick count turtles with [adopted? = true]

end

to set-pers-one ; turtle process
  set pers-one ( random-normal avg-pers-one 20 )      ; randomly assigns the the likelihood that a turtle will try to persuade another turtle at a tick
  if pers-one > 100 [ set pers-one 100 ]              ; the avg-pers-one slider sets the mean such that a higher setting means that more of the turtles will want to persuade
  if pers-one < 0 [ set pers-one 0 ]
end

to set-attitude ; turtle process
  set attitude ( random-normal avg-attitude 1 )       ; sets up turtles' start attitude, below the lowest adoption threshold and at least 1 like a 1-7 likert scale
  if attitude > 3.9 [ set attitude 3.9 ]
  if attitude < 1.0 [ set attitude 1.0 ]

end

to set-pers-power ; turtle process
  set pers-power ( random-normal avg-pers-power .1 )
  if pers-power > 2 [ set pers-power 2 ]
  if pers-power < 0 [ set pers-power 0 ]
end

to recolor
  ifelse adopted? = true [
    set color red ] [
    set color white ]
end

to update-friends ; turtle process
  set adopt-friends count link-neighbors with [ adopted? = true ]    ; updates number of adopted friends to use in the social influence calculation
  set non-adopt-friends count link-neighbors with [ adopted? = false ]
end

;next 4 reporters are created to record values of agents seeded with adoption

to-report init-degree
  report initial-degree
end

to-report init-between
  report initial-between
end

to-report init-pers-power
  report initial-pers-power
end

to-report init-pers-likely
  report initial-pers-likely
end

;next four reporters report the agent values at each tick

to-report report-betweenness                                         ; creates the data for the monitor labeled betweenness centrality of adopted
  let adopted-betweenness mean ( [ nw:betweenness-centrality ] of turtles with [ adopted? = true ] )
  report adopted-betweenness
end

to-report report-betweenness-all                                         ; creates the data for the monitor labeled betweenness centrality
  let betweenness-all mean ( [ nw:betweenness-centrality ] of turtles)
  report betweenness-all
end

to-report report-degree                                              ; creates the data for the monitor labeled degree centrality of adopted
  let adopted-degree mean ( [ count link-neighbors ] of turtles with [ adopted? = true ] )
  report adopted-degree
end

to-report report-degree-all                                             ; creates the data for the monitor labeled degree centrality
  let degree-all mean ( [ count link-neighbors ] of turtles)
  report degree-all
end

to-report report-pers-power
  let adopt-pers-power mean [pers-power] of turtles with [adopted? = true]
  report adopt-pers-power
end

to-report pers-likely-adopted
  let adopt-pers-likely mean [pers-one] of turtles with [adopted? = true]
  report adopt-pers-likely

end

to-report global-clustering-coefficient
  let closed-triplets sum [ nw:clustering-coefficient * count my-links * (count my-links - 1) ] of turtles
  let triplets sum [ count my-links * (count my-links - 1) ] of turtles
  report closed-triplets / triplets
end

to go
  let status-quo-before? status-quo?
  set status-quo? true
  ask turtles [
    update-friends                                             ; friends updated at the start of each tick
    if adopted? = true and non-adopt-friends > 0 [ persuade ]  ; persuasive attempts are only made by adopters
    if adopted? = false [ update-attitude ] ]                  ; attitude change only happens among non-adopters
  ifelse status-quo-before? = true and status-quo? = true [
    set status-quo-counter status-quo-counter + 1 ] [
    set status-quo-counter 0 ]
  tick
  if ticks < 1001 [set adopted-at-tick sentence (adopted-at-tick) (count turtles with [adopted? = true])]
  if status-quo-counter = 50 [ stop ]
  if ( not any? turtles with [ not adopted? ] ) [ stop ]     ; if the go is switched to run forever, it will stop if the whole network has adopted
end

to persuade ; turtle process
  if ( random 100 < pers-one ) [
    ask one-of link-neighbors with [ adopted? = false ] [
      set message [ pers-power ] of myself ] ]               ; target will keep track of pers-power as message, to be used in update-attitude
end

to update-attitude ; turtle process
  let old-attitude attitude                                               ; sets up the check for attitude changes this tick
  set attitude attitude + message + ( adopt-friends * social-influence )  ; attitude will change based on most recent message received AND social influence
  if attitude > 7 [
    set attitude 7 ]                                                      ; make sure attitude can't go over max for Likert scale
  if attitude >= adopt-threshold [ set adopted? true ]                    ; any turtle that reached the threshold will be set to adopted
  recolor                                                                 ; updates color based on adoption
  set message 0                                                           ; resets message variable to 0
  if old-attitude != attitude [ set status-quo? false ]                   ; indicates if there has been any attitude change this tick
end
@#$#@#$#@
GRAPHICS-WINDOW
407
121
775
490
-1
-1
10.91
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

SLIDER
3
11
220
44
num-agents
num-agents
0
500
100.0
1
1
NIL
HORIZONTAL

BUTTON
73
171
136
204
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
5
171
67
204
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
3
216
203
383
Adoptions over time
Time
Adoptions
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot count turtles with [adopted?]"

MONITOR
284
168
390
213
Number Adopted
count turtles with [adopted?]
17
1
11

SLIDER
3
89
220
122
percent-adopted-start
percent-adopted-start
0
1
0.05
.01
1
NIL
HORIZONTAL

SLIDER
3
51
220
84
min-links
min-links
1
num-agents - 1
2.0
1
1
NIL
HORIZONTAL

MONITOR
214
269
391
314
Betweeness Centrality of Adopted
report-betweenness
2
1
11

SLIDER
226
12
398
45
avg-attitude
avg-attitude
1
3.9
3.0
.1
1
NIL
HORIZONTAL

SLIDER
227
52
399
85
adopt-threshold
adopt-threshold
4
7
7.0
.1
1
NIL
HORIZONTAL

SLIDER
227
90
399
123
avg-pers-power
avg-pers-power
0
2
0.9
.05
1
NIL
HORIZONTAL

SLIDER
227
129
399
162
social-influence
social-influence
0
.5
0.0
.005
1
NIL
HORIZONTAL

SLIDER
4
129
220
162
avg-pers-one
avg-pers-one
0
100
10.0
1
1
NIL
HORIZONTAL

BUTTON
142
171
232
204
go forever
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
213
218
391
263
Degree Centrality of Adopted
report-degree
2
1
11

MONITOR
214
320
391
365
Persuasive Power of Adopted
report-pers-power
1
1
11

MONITOR
214
371
391
416
Persuasion Likelihood of Adopted
pers-likely-adopted
1
1
11

MONITOR
403
64
596
109
 Degree Centrality of Seeded (mean)
init-degree
1
1
11

MONITOR
604
64
816
109
Betweenness Centrality of Seeded (mean)
init-between
1
1
11

MONITOR
5
437
193
482
Persuasive Power of Seeded
init-pers-power
1
1
11

MONITOR
4
387
193
432
Persuasion Likelihood of Seeded
init-pers-likely
1
1
11

MONITOR
403
13
596
58
Degree Centrality (mean)
report-degree-all
2
1
11

MONITOR
604
13
816
58
Betweeness Centrality (mean)
report-betweenness-all
2
1
11

MONITOR
214
421
391
466
Network Clustering Coefficient
global-clustering-coefficient
2
1
11

@#$#@#$#@
## WHAT IS IT?

This model attempts to operationalize the basic ideas of the Boster et al. (2012) superdiffuser model while expanding the model by identifying moderators. That model argues that diffusion of a behavior through a network will be accelerated if the agents who are seeded with the behavior are connectors, persuaders, and mavens. Connectors have many connections and span boundaries between groups. Persuaders have heightened ability to change the attitudes of others by effectively adapting their arguments. Mavens are people who are highly involved in a given topic and enjoy educating others about that topic.

These three traits are represented in the model. Connectors are represented by the extent to which the  agents have high degree and betweenness centrality. Persuaders are represented by the extent to which agents are able to produce large attitude changes in those they try to persuade. Mavens are represented by their likelihood of trying to persuade another agent they are connected to at each tick.

The other variables represent aspects of the agents and the context that can alter both the speed of diffusion as well as the importance of the various superdiffuser traits of the seeded agents.

## HOW IT WORKS

At the start, a set proportion of the members of the network have adopted the behavior. Each of them occupies a different place in the network and thus they have varying levels of degree and betweenness centrality. Each of them also has a persuasive ability such that when they try to persuade other agents they are tied to, they increase the positivity of the target agent's attitude to a degree determined by the persuading agent's persuasive ability. Each also has a likelihood of trying to persuade one of the agents they are tied to at each tick such that they may have a zero percent chance up to a 100 percent chance.

At start, a scale-free network is created such that many of the agents have the minumum number of links and a few have many links, creating a hierarchy. The hierarchy is flattened to the extent to which the minimum number of links is increased. Some number of agents are seeded with having adopted the behavior and their color is set to pink.

At each tick, each agent that has not adopted the behavior gains a small increase in attitude through basic imitation based on how many of the agents they are linked to has adopted and the social influence strength. If social-influence is set to zero, then this does not occur.

At each tick, any agent that has raised their attitude above the threshold for adoption, adopts the behavior.

At each tick, any agent that has adopted the behavior may try to persuade one of the agents they are connected to and if they do they increase that neighbor's attitude by the amount the persuading agent is able.

Agents who have adopted turn from white to red.

If either all agents have adopted or attitudes have not changed for 50 ticks, the run stops.

## HOW TO USE IT

num-agents defines how many people will be in the network. 

min-links defines the minimum number of ties each person in the network will have. If it is set at 1 then each person is connected to at least one person. If it is set at 5 then each person is connected to at least five others. It is set up to be a scale-free network so some of the people will have substantially more connections than this minimum.

percent-adopted-start defines what proportion of the network will start the simulation having already adopted the behavior. They are shown in pink at setup.

avg-pers-one is the average of the likelihood that each turtle who has adopted will decide to try to persuade another turtle at each tick.

avg-attitude is the average of the attitudes of the turtles at setup. It is bounded by 1 and 3.9, ranging from the minumum to just below the midpoint of a 1-7 attitude scale. 

adopt-threshold is the attitude score at which a turtle who was not seeded with adoption will adopt the behavior.

avg-pers-power is the average of the agents persuasive ability. Persuasive ability is the size of the increase in the attitude of an agent they try to persuade.

social-influence is the size of the increase in an agent's attitude who has not adopted they get from each agent they are connected to that has already adopted. Social influence is assumed to be only positive such that agents who have not adopted do not negatively influence other agents' attitudes.

Setup creates a scale-free network in which the lines indicate undirected ties between the people. At the start, those who have already adopted are shown in pink and those who have not are in white.

Go runs a tick. During each tick, people who have not adopted do nothing except increase their attitudes from the social influence effect and/or in response to a persuasive attempt. Those who have adopted may or may not increase the attitude of one of their neighbors who has not already adopted. When agents adopt the behavior, they turn red.

Adoptions Over Time indicates how many people in the network have adopted over time to show how steep the adoption curve is in this run.

Number Adopted shows how many agents total have adopted at each tick.

Degree Centrality of Adopted shows the average degree centrality score for those who have adopted the behavior (degree centrality is the number of agents an agent is connected to). The start value at Setup is the average degree centrality of those assigned to have adopted at start. At each tick it is recalculated based on the degree centrality of all of those who have adopted. 

Betweenness Centrality of Adopted shows the average betweenness centrality score for those who have adopted the behavior (betweenness centrality is the extent to which a particular person is the link between otherwise unconnected groups). The start value at Setup is the average betweenness centrality of those assigned to have adopted at start. At each tick it is recalculated based on the betweenness centrality of all of those who have adopted. 

Network Clustering Coefficient displays a property of the network as a whole after Setup. It indicates an aspect of the Netlogo nw:clustering-coefficient such that the global clustering coefficient is calculated and reported. It indicates the extent to which people are more likely to be integrated together in the network or, if high, the extent to which the network is composed of separate groups with fewer ties among them.

## THINGS TO NOTICE

Who starts out pink can influence how rapidly and even if full diffusion happens. If the likelihood people wish to persuade is low, it is possible that those who start with the adopted behavior will not want to persuade very often. Also, when the number of minimum ties is low, there will be occasions where complete diffusion does not happen unless the social influence slider is above zero. If someone only has one connection and that person is not motivated to persuade anyone, that person may never adopt. You can usually see them at the periphery of the network. 

## THINGS TO TRY

min-links can create rapid diffusion when increased. If everyone has at least 10 friends, the odds are very good that at least one of them will have adopted and want to persuade any given person.

Increasing adopt-threshold can slow down diffusion and so can avg-pers-power.

Try combining different high and low values to see if they trade off in the speed with which total diffusion occurs. For example, high social influence can make up for low avg-pers-power or avg-pers-one.


## EXTENDING THE MODEL

We may try adding negative social influence or even negative persuaders who actively try to persuade people negatively. Another alternative is to give people the chance to try to persuade more than one person per tick rather than varying the likelihood they try to persuade one of their neighbors. We may also try different setup network structures besides the scale-free network.

## NETLOGO FEATURES



## RELATED MODELS

Communication-T-T, Network Example, Spread of Disease

## CREDITS AND REFERENCES

Created by Christopher J. Carpenter, Shannon Cruz, & Reed Reynolds
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="ClusteringRun" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>ticks</metric>
    <metric>count turtles with [adopted?]</metric>
    <metric>init-degree</metric>
    <metric>init-between</metric>
    <metric>init-pers-power</metric>
    <metric>init-pers-likely</metric>
    <metric>global-clustering-coefficient</metric>
    <enumeratedValueSet variable="percent-adopted-start">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-agents">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-pers-power">
      <value value="0.05"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-influence">
      <value value="0"/>
      <value value="0.05"/>
      <value value="0.1"/>
      <value value="0.15"/>
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-links">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adopt-threshold">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-attitude">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="3.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-pers-one">
      <value value="0"/>
      <value value="10"/>
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experimenttest" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>ticks</metric>
    <metric>count turtles with [adopted?]</metric>
    <metric>init-degree</metric>
    <metric>init-between</metric>
    <metric>init-pers-power</metric>
    <metric>init-pers-likely</metric>
    <metric>adopted-at-tick</metric>
    <enumeratedValueSet variable="percent-adopted-start">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-agents">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-pers-power">
      <value value="0.9"/>
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-influence">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-links">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-attitude">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adopt-threshold">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-pers-one">
      <value value="10"/>
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experimenttest 3" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <metric>ticks</metric>
    <metric>count turtles with [adopted?]</metric>
    <metric>init-degree</metric>
    <metric>init-between</metric>
    <metric>init-pers-power</metric>
    <metric>init-pers-likely</metric>
    <metric>adopted-at-tick</metric>
    <enumeratedValueSet variable="percent-adopted-start">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-agents">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-pers-power">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-influence">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-links">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-attitude">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adopt-threshold">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-pers-one">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
