;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; GPL 3 license                                                                ;
;    This program is free software: you can redistribute it and/or modify      ;
;    it under the terms of the GNU General Public License as published by      ;
;    the Free Software Foundation, either version 3 of the License, or         ;
;    (at your option) any later version.                                       ;
;                                                                              ;
;    This program is distributed in the hope that it will be useful,           ;
;    but WITHOUT ANY WARRANTY; without even the implied warranty of            ;
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             ;
;    GNU General Public License for more details.                              ;
;                                                                              ;
;    You should have received a copy of the GNU General Public License         ;
;    along with this program.  If not, see <https://www.gnu.org/licenses/>.    ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
extensions [ csv table ]

globals [ this-seed links-list ]

breed [ resources resource ]
breed [ consumers consumer ]

consumers-own [ location target-location resource-requirement resource-stock metabolism active? spawn-energy consumption-rate ]
resources-own [ subnet-id node-id current-supply resource-capacity regrow-rate ]

to setup
  clear-all
  ; Set a random seed unless one is provided
  ifelse my-seed != 0 [ set this-seed my-seed ] [ set this-seed new-seed ]
  random-seed this-seed
  import-node-attributes
  import-links
  make-consumers
  reset-ticks
  ; Record initial state of model
  record-initial-consumer-state
  record-initial-resource-state
end

to go
  if not any? consumers with [ active? = true ] [ stop ]
  ; All resources regrow by regrow-rate each tick (up to resource-capacity)
  regrow-resource
  ; Consumers wander randomly and consume resources
  consume-resource
  ; Consumers with enough energy spawn new consumers
  spawn-consumers
  ; All consumers check and consume basal metabolism from stocks
  check-supplies
  tick
  ; Record data - not currently done every tick or the file size quickly expands
  if (ticks = 10 or ticks = 100 or ticks = 250 or ticks = 500 or ticks = 750 or ticks = 1000) [
    write-consumer-state
  ]
  write-resource-state
end

; Observer procedure to check if consumers have enough energy to meet basal metabolic requirements
to check-supplies
  ask consumers with [ active? = true ] [
    ; If they don't have enough energy, they become inactive
    ifelse resource-stock <= resource-requirement [
      become-inactive
    ] [
      ; Otherwise, metabolise basal requirement from resource stocks
      set resource-stock resource-stock - resource-requirement
    ]
  ]
end

; Consumer procedure to set consumers without enough energy as 'inactive' - effectively dead, but
; still in the system to allow for expansion/future work
to become-inactive ; consumer procedure
  set resource-stock 0
  set active? false
  set hidden? true
end

; Observer procedure to handle consumers eating resources - if there is nothing for them to eat, they move on, otherwise they consume
; what is available (up to their maximum consumption-rate per timestep)
to consume-resource
  ask consumers with [ active? = true ] [
    ifelse resource-stock < 0 [
      ; If consumer has used up all energy getting to the resource and is in negative resource stocks, it dies (but not if it is at 0)
      become-inactive
    ] [
      ; If there is nothing to eat here, consumers move on
      if not any? resources-here with [ current-supply > 0 ] [ move-on ]
    ]
  ]
  ; Only check resources with something for consumers to take, and that have consumers on them
  ask resources with [ current-supply > 0 ] [
    if any? consumers-here with [ active? = true ] [
      let active-consumers-here consumers-here with [ active? = true ]
      let total-resource-req sum [ consumption-rate ] of active-consumers-here
      ifelse total-resource-req <= current-supply [
        ; Let everyone take what they want for this tick
        ask active-consumers-here [
          set resource-stock resource-stock + consumption-rate
        ]
        ; Deplete resource
        set current-supply current-supply - total-resource-req
      ] [
        ; If there is not enough for everyone to take their fill, have consumers split what's available and move on
        let split-supply current-supply / count active-consumers-here
        ask active-consumers-here [
          set resource-stock resource-stock + split-supply
          move-on
        ]
        ; Drain resource
        set current-supply 0
      ]
    ]
  ]
end

; Consumer procedure for moving toward a previously-identified target resource, or identifying a new target resource
to move-on
  ifelse patch-here != [ patch-here ] of target-location [
    ifelse distance target-location <= 1 [
      ; If consumer is right next to the target, move to it
      let small-step (distance target-location)
      face target-location
      move-to target-location
      let this-link link [ who ] of location [ who ] of target-location
      ; Energy consumed is proportional to distance travelled
      set resource-stock ( resource-stock - ( metabolism * small-step ) )
    ] [
      ; If >1 unit of distance to go, keep moving toward target
      face target-location
      fd 1
      let this-link link [ who ] of location [ who ] of target-location
      set resource-stock ( resource-stock - metabolism )
    ]
  ] [
    ; If arrived, update current location to (old) target location
    set location target-location
    ; Identify new target and move toward it, if there are any to move toward
    if count [ link-neighbors ] of location > 0 [
      set target-location one-of [ link-neighbors ] of location
      face target-location
      fd 1
      set resource-stock (resource-stock - metabolism)
    ]
  ]
end

; Observer procedure to identify resources below maximum capacity and have them regrow
to regrow-resource
  ask resources with [ current-supply < resource-capacity ] [
    ifelse current-supply + regrow-rate <= resource-capacity [
      ; Resources fill up by regrow-rate
      set current-supply current-supply + regrow-rate
    ] [
      ; Resources that are nearly at full capacity fill up to full capacity, but not beyond that
      set current-supply resource-capacity
    ]
  ]
end

; Observer procedure to identify consumers who are able to spawn, and have them reproduce
to spawn-consumers
  ask consumers with [ resource-stock > ( spawn-energy * 2 ) ] [
    ; Spawn a new consumer, set initial resource stock to the amount it took to spawn
    hatch-consumers 1 [ set resource-stock spawn-energy ]
    ; Deplete parent's resource stock by the amount it took to spawn
    set resource-stock resource-stock - spawn-energy
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                         ;
;             Setup functions             ;
;                                         ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Observer procedure to read in a file that contains resource node-specific attributes,
; including an unique identification number, and creates the resources
to import-node-attributes
  let i 1
  file-open nodes-file
  ; Skip headers
  let headers csv:from-row file-read-line
  ; Data should be in this order: network-id xcor ycor node-id
  ; Network-id stores the subnetwork to which the node belongs - not currently used
  ; Node-id is just an incremented index
  while [not file-at-end?]
  [
    let items csv:from-row file-read-line
    create-resources 1 [
      set subnet-id item 0 items
      set xcor item 1 items
      set ycor item 2 items
      set node-id item 3 items
      set shape "square"
      set color yellow
      set resource-capacity random max-resource-capacity
      set regrow-rate random max-resource-regrow-rate
      set current-supply resource-capacity
    ]
  ]
  file-close
end

; Observer procedure to read in a file that contains all the links.
; The first column contains the node-id of the resource originating the link.
; The second column is the node-id of the resource on the other end of the link.
; Links are not directed currently, so it doesn't make a difference which node is on which end.
to import-links
  file-open links-file
  let headers csv:from-row file-read-line
  while [not file-at-end?]
  [
    let items csv:from-row file-read-line
    ask get-node (item 0 items)
    [
      create-link-with get-node (item 1 items)
    ]
  ]
  file-close
end

;; Helper procedure for looking up a node by node-id.
to-report get-node [id]
  report one-of resources with [node-id = id]
end

; Observer procedure to create consumers
to make-consumers
  create-consumers n-consumers [
    set color red
    set shape "bug"
    set active? true
    set resource-requirement consumer-resource-requirement
    set resource-stock initial-resource-owned
    set metabolism consumer-metabolism
    set consumption-rate consumer-consumption-rate
    set spawn-energy consumer-spawn-energy
    set location one-of resources
    set target-location location
    move-to location
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                         ;
;       Data collection reporters         ;
;                                         ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Report mean resource supply across consumers - for plotting and data collection
to-report mean-consumer-resource-stock
  ifelse count consumers with [ active? = true ] > 0 [
    report mean [ resource-stock ] of consumers with [ active? = true ]
  ] [
    report 0
  ]
end

; Report standard deviation of resource supply across consumers - for plotting and data collection
to-report sd-consumer-resource-stock
  ifelse count consumers with [ active? = true ] > 3 [
    report standard-deviation [ resource-stock ] of consumers with [ active? = true ]
  ] [
    report "NA"
  ]
end

; Report Gini coefficient of resource supply across consumers - for plotting and data collection
;; Adapted from Wilensky, U. (1998). NetLogo Wealth Distribution model.
to-report gini-consumer-resource-stock
  let active-consumers count consumers with [ active? = true ]
  let sorted-res-stocks sort [ resource-stock ] of consumers with [ active? = true ]
  let total-res-stock sum sorted-res-stocks
  let res-stock-sum-so-far 0
  let index 0
  let gini-index-sum 0

  ; Calculate the Gini index
  repeat active-consumers [
    set res-stock-sum-so-far (res-stock-sum-so-far + item index sorted-res-stocks)
    set index (index + 1)
    set gini-index-sum
      gini-index-sum +
      (index / active-consumers) -
      (res-stock-sum-so-far / total-res-stock)
  ]
  report (gini-index-sum / active-consumers / 0.5)
end

; Report count of active (alive) consumers - for plotting and data collection
to-report count-active-consumers
  report (count consumers with [ active? = true ])
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                         ;
;           Recording functions           ;
;                                         ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Observer procedure to record model's initial state - writes file headers and consumer data.
; NOTE: This subroutine will delete an existing file with the same name! If you are running the code locally,
; be sure to use different seeds to save the files with unique names if you don't want them to be overwritten.
to record-initial-consumer-state
  let consumer-output-file-bspace ""
  ifelse empty? behaviorspace-experiment-name [
    set consumer-output-file-bspace (word "x-1-" this-seed "_consumerdata.csv")
  ] [
    set consumer-output-file-bspace (word "x-" behaviorspace-experiment-name "_" this-seed "_" behaviorspace-run-number "_consumerdata.csv")
  ]
  if file-exists? consumer-output-file-bspace [
    file-delete consumer-output-file-bspace
  ]
  file-open consumer-output-file-bspace
  file-print "ticks,consumer,is-active,x,y,resource-stock"
  ask consumers [
    file-print (word ticks "," who "," active? "," xcor "," ycor "," resource-stock)
  ]
  file-close
end

; Observer procedure to record model's initial state - writes file headers and resource data.
; NOTE: This subroutine will delete an existing file with the same name! If you are running the code locally,
; be sure to use different seeds to save the files with unique names if you don't want them to be overwritten.
to record-initial-resource-state
  let resource-output-file-bspace ""
  ifelse empty? behaviorspace-experiment-name [
    set resource-output-file-bspace (word "x-1-" this-seed "_resourcedata.csv")
  ] [
    set resource-output-file-bspace (word "x-" behaviorspace-experiment-name "_" this-seed "_" behaviorspace-run-number "_resourcedata.csv")
  ]
  if file-exists? resource-output-file-bspace [
    file-delete resource-output-file-bspace
  ]
  file-open resource-output-file-bspace
  file-print "ticks,resource,xcor,ycor,current-supply,resource-capacity,regrow-rate"
  ask resources [
    file-print (word ticks "," who "," xcor "," ycor "," current-supply "," resource-capacity "," regrow-rate)
  ]
  file-close
end

; Observer procedure to record consumer location, status, and resource stock.
to write-consumer-state
  let consumer-output-file-bspace ""
  ifelse empty? behaviorspace-experiment-name [
    set consumer-output-file-bspace (word "x-1-" this-seed "_consumerdata.csv")
  ] [
    set consumer-output-file-bspace (word "x-" behaviorspace-experiment-name "_" this-seed "_" behaviorspace-run-number "_consumerdata.csv")
  ]
  file-open consumer-output-file-bspace
  ask consumers [
    file-print (word ticks "," who "," active? "," xcor "," ycor "," resource-stock)
  ]
  file-close
end

; Observer procedure to record resource location, current supply, maximum supply, and regrowth rate
to write-resource-state
  let resource-output-file-bspace ""
  ifelse empty? behaviorspace-experiment-name [
    set resource-output-file-bspace (word "x-1-" this-seed "_resourcedata.csv")
  ] [
    set resource-output-file-bspace (word "x-" behaviorspace-experiment-name "_" this-seed "_" behaviorspace-run-number "_resourcedata.csv")
  ]
  file-open resource-output-file-bspace
  ask resources [
    file-print (word ticks "," who "," xcor "," ycor "," current-supply "," resource-capacity "," regrow-rate)
  ]
  file-close
end
@#$#@#$#@
GRAPHICS-WINDOW
255
40
983
625
-1
-1
1.43713
1
10
1
1
1
0
1
1
1
0
500
0
400
1
1
1
ticks
30.0

BUTTON
25
40
88
73
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
95
40
158
73
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
25
85
240
118
n-consumers
n-consumers
100
1000
100.0
1
1
NIL
HORIZONTAL

SLIDER
25
125
240
158
max-resource-capacity
max-resource-capacity
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
25
165
240
198
consumer-resource-requirement
consumer-resource-requirement
1
100
1.0
1
1
NIL
HORIZONTAL

SLIDER
25
245
240
278
initial-resource-owned
initial-resource-owned
0
100
30.0
1
1
NIL
HORIZONTAL

PLOT
25
445
240
625
Mean consumer resource stock
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean-consumer-resource-stock"

BUTTON
165
40
242
73
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
25
285
240
318
max-resistance
max-resistance
1
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
25
325
240
358
consumer-metabolism
consumer-metabolism
0
100
1.0
1
1
NIL
HORIZONTAL

SLIDER
25
205
240
238
max-resource-regrow-rate
max-resource-regrow-rate
1
100
5.0
1
1
NIL
HORIZONTAL

SLIDER
25
365
240
398
consumer-consumption-rate
consumer-consumption-rate
1
100
5.0
1
1
NIL
HORIZONTAL

SLIDER
25
405
240
438
consumer-spawn-energy
consumer-spawn-energy
1
100
50.0
1
1
NIL
HORIZONTAL

PLOT
1005
40
1205
190
Active consumers
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot count consumers with [ active? = true ]"

INPUTBOX
1005
195
1222
255
nodes-file
NIL
1
0
String

INPUTBOX
1005
260
1222
320
links-file
NIL
1
0
String

INPUTBOX
1005
325
1157
385
my-seed
0.0
1
0
Number

@#$#@#$#@
## WHAT IS IT?

This model was designed to explore the emergent heterogeneity in a population of generic consumers, who forage for energy resources in a network. In the original model, the networks were soil macropore networks generated from images taken of soil profiles.

## HOW IT WORKS

The agents (consumers) are random walkers who navigate from resource node to resource node, moving one step per tick. At each timestep, they require a certain amount of energy (basal metabolism), plus energy required for movement (active metabolism), if they are moving. Consumers can also choose to remain on a resource node if it has enough energy for them to do so. If a consumer has more than twice their spawn energy requirement, they spawn a new consumer identical to themselves, who inherits half of the parent's energy reserves. Consumers who run out of energy become inactive (die). 

## HOW TO USE IT

Specify two CSV files in the 'nodes-file' and 'links-file' inputs - the nodes file should include headings and be formatted as network_ID, x, y, node_ID. The links file should include the headings node_from, node_to. Currently, the network_ID in the nodes file is not used, but it was included in planning for future work.

Set the sliders on the right of the viewer to the values of interest. If needed, a seed can be set in the 'my-seed' input to recreate the stochastic elements of the run. Otherwise, set this to 0 to allow Netlogo to choose a random seed.

Setup reads in the nodes and links and creates the network - this may take several seconds for large networks. The resources are placed at the nodes. Go and go-once behave as expected, running the simulation indefinitely (or until all consumers have become inactive) or for a single time step, respectively.

## THINGS TO NOTICE

Where does the population tend to increase the most? How does the presence of disconnected subnetworks influence the population growth and dynamics?

## THINGS TO TRY

Set up some plots and look at the mean, SD or distribution of energy reserves in the population. Does the population become more or less heterogeneous over time?

## EXTENDING THE MODEL

The model could easily be extended to include heterogeneous agents (such as by implementing a mean and SD for the metabolism rates, spawn energy threshold, etc.), predation, mating/sexual reproduction, or by allowing the consumers to add to or modify the network.

## NETLOGO FEATURES

N/A

## RELATED MODELS

N/A

## CREDITS AND REFERENCES

The calculation of the Gini coefficient was adapted from Wilensky, U. (1998). NetLogo Wealth Distribution model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="x-1" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2000"/>
    <exitCondition>count consumers with [ active? = true ] = 0</exitCondition>
    <metric>count-active-consumers</metric>
    <metric>mean-consumer-resource-stock</metric>
    <metric>sd-consumer-resource-stock</metric>
    <metric>entropy-consumer-resource-stock</metric>
    <metric>gini-consumer-resource-stock</metric>
    <enumeratedValueSet variable="consumer-resource-requirement">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-metabolism">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-consumption-rate">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-spawn-energy">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-resource-capacity">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-resource-regrow-rate">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-resource-owned">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-resistance">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="nodes-file">
      <value value="&quot;soil_image_analysis/abm_networks/4835_nodes.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="links-file">
      <value value="&quot;soil_image_analysis/abm_networks/4835_links.csv&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="test-exp" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <final>write-consumer-state</final>
    <timeLimit steps="100"/>
    <exitCondition>count consumers with [ active? = true ] = 0</exitCondition>
    <metric>mean-consumer-resource-stock</metric>
    <metric>sd-consumer-resource-stock</metric>
    <metric>count consumers with [ active? = true ]</metric>
    <enumeratedValueSet variable="links-file">
      <value value="&quot;soil_image_analysis/abm_networks/4835_links.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-resource-capacity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-metabolism">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-resource-regrow-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-resistance">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-resource-owned">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-spawn-energy">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-resource-requirement">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="nodes-file">
      <value value="&quot;soil_image_analysis/abm_networks/4835_nodes.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="consumer-consumption-rate">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
