extensions [nw] ;; This is the netlogo network extension called from the code
directed-link-breed [ directed-edges directed-edge ] ;; category of link. Only used for centrality measures
undirected-link-breed [ undirected-edges undirected-edge ] ;; category of link. Only used for centrality measures

Globals ;; global variables (these are the observer-level variables)

[
  Signal
  Iteration-count
 ]


turtles-own ;; Potential agent states
[
  susceptible?        ;; if true, the turtle will accept the new environmental signal and can be updated
  updated?            ;; if true, the turtle has been updated with the new environmental signal
  agreed?             ;; if true, the turtle has a shared problem-solving schema
  implemented?        ;; if true, the turtle has implemented the agreed problem-solving action
  reviewed?           ;; if true, the turtle has reviewed the problem solving action
  adjusted?           ;; if true, the turtle has adjusted the problem solving action
  resistant?          ;; if true, the turtle will resist/reject the new signal and can't be updated
  schema-check-timer  ;; number of ticks since this turtle's last schema-check
  ]

to setup
  clear-all
   if Network-source = "Generate" ;; This 'if' condition prevents accidently generating a network over the top of an imported network
  [create-network]
    set-current-plot "degree distribution"
   set signal (Signal-base + (0))
  set-default-shape turtles "person"
     reset-ticks
 end

to cycle
  ask turtles
 [ set color blue
  set adjusted? false ;;one of two end states from previous cycle
  set susceptible? true
  set resistant? false ;;one of two end states from previous cycle
  become-susceptible
  set schema-check-timer random schema-check-frequency
    ]
   set Iteration-count Iteration-count + 1 ;; tracking 1 iteration
  set-current-plot "Network Status"
    ask n-of Initial-detection-size turtles ;; this sets the initial update size
    [ become-updated ]
  ask links [ set color yellow ]
end

;; Load network ;;Only used if importing a network - not used if generating the network
to load-network
  nw:set-context turtles links
  nw:load-graphml "CQINTest1.graphml" ;;target graphml file must be in C:\Users\bill\OneDrive\Documents\NetLogo ;; (i.e. must be the same folder as the model)
   [
  set size 3
       set shape "person"
    ;; set color blue
    set updated? false
    set susceptible? true
    set resistant? false
     become-susceptible
    set schema-check-timer random schema-check-frequency
  ]
  reset-ticks
  end

;;Generate networks rather than use imported network
to create-network ;; create the agent network - random or preferential attachment
  if network-type = "random"
 [setup-nodes
    setup-spatially-clustered-network ]
   if network-type = "preferential-attachment" [create-preferential-attachment ]
end

to setup-nodes
  set-default-shape turtles "person"
    create-turtles number-of-nodes
 [
    ; for visual reasons, we don't put any nodes *too* close to the edges
    setxy (random-xcor * 0.9) (random-ycor * 0.9)
    set size 1.5
    become-susceptible
   set schema-check-timer random schema-check-frequency
 ]
end

to create-preferential-attachment ;; this is the Barabasi-Albert method of creating a preferential attachment graph
    nw:generate-preferential-attachment turtles links (number-of-nodes) (node-degree) ;; N.B for this  preferential attachment code in Netlogo the node degree represents the MINIMUM degree
  [
    set size 2
    set shape "person"
    set color blue
    set updated? false
    set susceptible? true
    set resistant? false
    become-susceptible
    set schema-check-timer random schema-check-frequency
      ]
end

to create-random-network ;; generate an Erdos-Renyi random graph
    nw:generate-random turtles links (number-of-nodes) (number-of-nodes * node-degree / 2  * 0.5) [
    set size 1.5
    set shape "person"
    set color blue
    set updated? false
    set susceptible? true
    set resistant? false
  ]
end


to layout ;; visually set out network
 repeat 20 [
;; layout-spring turtles links 1 22 4
   layout-spring turtles links 0.7 20 3
;; layout-radial turtles links (turtle 0)
;; layout-circle turtles 18
;; update-plots
   display
     ]
end

to setup-spatially-clustered-network
  let num-links (node-degree * number-of-nodes) / 2  ;; for random network node degree is average
  while [count links < num-links ]
  [
    ask one-of turtles
    [
      let choice (min-one-of (other turtles with [not link-neighbor? myself])
                   [distance myself])
      if choice != nobody [ create-link-with choice ]
    ]
  ]
  ; make the network look a little prettier
  repeat 10
  [
   layout-spring turtles links 0.3 (world-width / (sqrt number-of-nodes)) 1
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; set initial signal detection;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to trigger
;; Note that this can be repeated if desired
  set Iteration-count Iteration-count + 1 ;; tracking 1 iteration
  set-current-plot "Network Status"
     ask n-of Initial-detection-size turtles ;; this sets the initial update size
    [ become-updated ]
  ask links [ set color yellow ]
  reset-ticks
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Underlying process sequence (The Improvement cycle phases);;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go  ;; this is the first phase of the learning classifier cycle, detecting and assigning credit to a new environmental signal
  if count turtles with [resistant?] >= count turtles * max-non-co-operating
    [stop]
  if  all? turtles [updated? or resistant?]
    [stop]
      if Iteration-count = Planned-iterations + 1
  [stop]
     if Signal = Signal-target
  [stop ]
  ask turtles
  [
     set schema-check-timer schema-check-timer + 1
     if schema-check-timer >= schema-check-frequency
       [ set schema-check-timer 0 ]
  ]
  spread-signal
   do-schema-checks
    tick
;;  nw:save-graphml "example.graphml" ;; option saves generated network to graphml format for analysis outside netlogo
end

to go-2  ;; this is the second phase of the learning classifier cycle, determining and agreeing available problem-solving elements and actions
  if count turtles with [resistant?] >= count turtles * max-non-co-operating
  [stop]
  if all? turtles [agreed? or resistant?]
   [ stop ]
 if Iteration-count = Planned-iterations + 1
   [ stop ]
  if Signal = Signal-target
  [stop ]
  ask turtles
   [
     set schema-check-timer schema-check-timer + 2
    if schema-check-timer >= schema-check-frequency
       [ set schema-check-timer 0 ]
  ]
problem-solve
    do-schema-checks-2
  tick
end

to go-3  ;; this is the third phase of the learning classifier cycle, implementing problem-solving actions
  if count turtles with [resistant?] >= count turtles * max-non-co-operating
  [stop]
  if all? turtles [implemented? or resistant?]
   [ stop ]
  if Iteration-count = Planned-iterations + 1
  [stop]
   if Signal = Signal-target
  [stop ]
    ask turtles
   [
     set schema-check-timer schema-check-timer + 3
    if schema-check-timer >= schema-check-frequency
       [ set schema-check-timer 0 ]
  ]
 implement
  do-schema-checks-3
  tick
end

to go-4  ;; fourth phase of the learning classifier cycle, check/assess effectiveness, progress, unintended consequences
   if count turtles with [resistant?] >= count turtles * max-non-co-operating
  [stop]
  if all? turtles [reviewed? or resistant?]
   [ stop ]
 if Iteration-count = Planned-iterations + 1
  [stop]
   if Signal = Signal-target
  [stop ]
  ask turtles
   [
     set schema-check-timer schema-check-timer + 4
    if schema-check-timer >= schema-check-frequency
       [ set schema-check-timer 0 ]
  ]
 review
  do-schema-checks-4
  tick
end

to go-5  ;; fifth phase of the learning classifier cycle, adjusting (within intended parameters)
 if count turtles with [resistant?] >= count turtles * max-non-co-operating
  [stop]
  if all? turtles [adjusted? or resistant?]
   [ stop ]
  if Iteration-count = Planned-iterations + 1
  [stop]
   if Signal = Signal-target
  [stop ] if Signal = Signal-target
  [stop ]
    ask turtles
   [
     set schema-check-timer schema-check-timer + 5
    if schema-check-timer >= schema-check-frequency
       [ set schema-check-timer 0 ]
  ]
 adjust
  do-schema-checks-5
  tick
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; turtle procedures (behaviour and actions);;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to become-susceptible  ;; turtle procedure
  set susceptible? true
  set updated? false
  set agreed? false
  set implemented? false
  set reviewed? false
  set adjusted? false
  set resistant? false
  set color blue
end

to become-updated  ;; turtle procedure
  set updated? true
  set susceptible? false
  set agreed? false
  set implemented? false
  set reviewed? false
  set adjusted? false
  set resistant? false
  set color red
end

to become-agreed ;; turtle procedure
  set agreed? true
  set susceptible? false
  set updated? false
  set implemented? false
  set reviewed? false
  set adjusted? false
  set resistant? false
  set color green
end

to become-implemented  ;; turtle procedure
  set implemented? true
  set susceptible? false
  set updated? false
  set agreed? false
  set reviewed? false
  set adjusted? false
  set resistant? false
  set color brown
end

to become-reviewed  ;; turtle procedure
  set reviewed? true
  set susceptible? false
  set updated? false
  set agreed? false
  set implemented? false
  set adjusted? false
  set resistant? false
  set color pink
end

to become-adjusted  ;; turtle procedure
  set adjusted? true
  set susceptible? false
  set updated? false
  set agreed? false
  set implemented? false
  set reviewed? false
  set resistant? false
  set color orange
end

to become-resistant  ;; turtle procedure
  set susceptible? false
  set updated? false
  set agreed? false
  set implemented? false
  set reviewed? false
  set adjusted? false
  set resistant? true
  set color gray
  ask my-links [ set color gray - 2 ]
end


to spread-signal
  ask turtles with [updated?] ; n=initial detection size
    [ask link-neighbors with [susceptible? and not resistant?]
              [ if random-float 100  < (schema-share-effectiveness) * (Learning-gain ^ Iteration-count)
            [ become-updated ] ] ]
end

 to problem-solve
 ask turtles with [updated?]
  ;;ask turtles with [updated? or agreed?] ;;
     [ask link-neighbors with [updated? and not resistant?]
                 [ if random-float 100  < (schema-share-effectiveness) * (learning-gain ^ Iteration-count) * (problem-solving-effectiveness * (Learning-gain ^ Iteration-count)) / problem-space-complexity ;; Option 1: Rationale tbc
            [ become-agreed ] ] ]
end

 to implement
 ask turtles with [agreed?]
;;  ask turtles with [agreed? or implemented?]
      [ask link-neighbors with [agreed? and not resistant?]
                 [ if random-float 100  < (schema-share-effectiveness * (learning-gain ^ Iteration-count)) * ((problem-solving-effectiveness * (Learning-gain ^ Iteration-count)) / problem-space-complexity) * ((100 - coevolution-constraints) / 100) ;; Option 1: Rationale tbc
            [ become-implemented ] ] ]
end

 to review
 ask turtles with [implemented?]
 ;; ask turtles with [implemented? or reviewed?]
      [ask link-neighbors with [implemented? and not resistant?]
           [ if random-float 100  < (schema-share-effectiveness) * (Learning-gain ^ Iteration-count)
                         [ become-reviewed ] ] ]
end

to adjust
   ask turtles with [reviewed?]
 ;; ask turtles with [reviewed? or adjusted?]
      [ask link-neighbors with [reviewed? and not resistant?]
        [ if random-float 100  < (schema-share-effectiveness) * (learning-gain ^ Iteration-count) * (problem-solving-effectiveness * (Learning-gain ^ Iteration-count)) / problem-space-complexity * (100 - coevolution-constraints) / 100  ;;
             [ become-adjusted ] ] ]
end


to do-schema-checks
  ask turtles with [updated? and schema-check-timer = 0]
  [
    if random 100 < (rejection-threshold *(0.8 + (problem-space-complexity / 100 ))) ;; Rejection threshold hypothesised as lightly coupled to problem space complexity in this version
    [
    ifelse random 100 < (rejection-chance *(0.8 + (problem-space-complexity / 100 )))
        [become-resistant ]
        [become-updated]
     ]
  ]
end

to do-schema-checks-2
ask turtles with [updated? and schema-check-timer = 0]
;;  ask turtles with [agreed? or updated? and schema-check-timer = 0]
;; ask turtles with [not resistant? and schema-check-timer = 0]
  [
    if random 100 < (rejection-threshold *(0.8 + (problem-space-complexity / 100 )))
    [
    ifelse random 100 < (rejection-chance *(0.8 + (problem-space-complexity / 100 )))
        [become-resistant]
        [become-agreed]
     ]
  ]
end

to do-schema-checks-3
   ask turtles with [agreed? and schema-check-timer = 0]
;;  ask turtles with [implemented? or agreed? and schema-check-timer = 0]
 ;;  ask turtles with [not resistant? and schema-check-timer = 0]
  [
    if random 100 < (rejection-threshold *(0.8 + (problem-space-complexity / 100 )))
    [
    ifelse random 100 < (rejection-chance *(0.8 + (problem-space-complexity / 100 )))
        [become-resistant]
        [become-implemented]
     ]
  ]
end

to do-schema-checks-4
 ask turtles with [implemented? and schema-check-timer = 0]
;;  ask turtles with [reviewed? or implemented? and schema-check-timer = 0]
  ;;  ask turtles with [not resistant? and schema-check-timer = 0]
  [
    if random 100 < (rejection-threshold *(0.8 + (problem-space-complexity / 100 )))
    [
    ifelse random 100 < (rejection-chance *(0.8 + (problem-space-complexity / 100 )))
        [become-resistant]
        [become-reviewed]
     ]
  ]
end

to do-schema-checks-5
 ask turtles with [reviewed? and schema-check-timer = 0]
;; ask turtles with [adjusted? or reviewed? and schema-check-timer = 0]
  ;;  ask turtles with [not resistant? and schema-check-timer = 0]
  [
    if random 100 < (rejection-threshold *(0.8 + (problem-space-complexity / 100 )))
    [
    ifelse random 100 < (rejection-chance *(0.8 + (problem-space-complexity / 100 )))
        [become-resistant]
        [become-adjusted]
     ]
  ]
  if count turtles with [resistant?] < count turtles * max-non-co-operating
    [set signal (Signal-base + (Signal-increment * Iteration-count))]
show-signal
end

to show-signal
  output-show Signal
end

;;;;;;;;;;;;;;;;;;;;;;;;
;; Centrality measures;;
;;;;;;;;;;;;;;;;;;;;;;;;

;; This section of code is from the Netlogo model library - 'Extensions NW General' Examples

to-report get-links-to-use ;; Reports the link set corresponding to the value of the links-to-use combo box
 report ifelse-value links-to-use = "directed"
   [ directed-edges ]
    [ undirected-edges ]
end

to betweenness
  centrality [ -> nw:betweenness-centrality ]
end

to eigenvector
  centrality [ -> nw:eigenvector-centrality ]
end

to closeness
 centrality [ -> nw:closeness-centrality ]
end

; Takes a centrality measure as a reporter task, runs it for all nodes
; and set labels, sizes and colors of turtles to illustrate result
to centrality [ measure ]
  nw:set-context turtles links
 ask turtles [
    let res (runresult measure) ; run the task for the turtle
    ifelse is-number? res [
      set label precision res 2
     set size res ; this will be normalized later
    ]
   [ ; if the result is not a number, it is because eigenvector returned false (in the case of disconnected graphs
      set label res
     set size 1
   ]
 ]
  normalize-sizes-and-colors
end

; We want the size of the turtles to reflect their centrality, but different measures
; give different ranges of size, so we normalize the sizes according to the formula
; below. We then use the normalized sizes to pick an appropriate color.
to normalize-sizes-and-colors
  if count turtles > 0 [
    let sizes sort [ size ] of turtles ; initial sizes in increasing order
    let delta last sizes - first sizes ; difference between biggest and smallest
    ifelse delta = 0 [ ; if they are all the same size
      ask turtles [ set size 1 ]
    ]
    [ ; remap the size to a range between 0.5 and 2.5
      ask turtles [ set size ((size - first sizes) / delta) * 2 + 0.5 ]
    ]
    ask turtles [ set color scale-color red size 0 5 ] ; using a higher range max not to get too white...
  ]
end

;;;;;;;;;;;;;
;; Clusters;;
;;;;;;;;;;;;;

;; This section of code from the Netlogo model library - Extensions NW General Examples

; Colorizes each node according to the community it is part of
to community-detection
  nw:set-context turtles get-links-to-use
  color-clusters nw:louvain-communities
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Highlighting and coloring of clusters ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; This section of code from the Netlogo model library - Extensions NW General Examples

to color-clusters [ clusters ]
  ; reset all colors
  ask turtles [ set color gray ]
  ask links [ set color gray - 2 ]
  let n length clusters
  ; Generate a unique hue for each cluster
  let hues n-values n [ i -> (360 * i / n) ]

  ; loop through the clusters and colors zipped together
  (foreach clusters hues [ [cluster hue] ->
    ask cluster [ ; for each node in the cluster
                  ; give the node the color of its cluster
      set color hsb hue 100 100
      ; Color links contained in the cluster slightly darker than the cluster color
      ask my-links with [ member? other-end cluster ] [ set color hsb hue 100 75 ]
    ]
  ])
end

;; Bill Wilson Complex Quality Improvement Network (CQIN) Massey University Research Project 2020-2023

; Reference Model: 'Virus on a network'
; Copyright 2008 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
265
10
724
470
-1
-1
11.0
1
10
1
1
1
0
0
0
1
-20
20
-20
20
1
1
1
ticks
30.0

SLIDER
933
104
1125
137
rejection-chance
rejection-chance
0.0
50
10.0
1
1
%
HORIZONTAL

SLIDER
933
69
1125
102
rejection-threshold
rejection-threshold
0.0
100
25.0
1
1
%
HORIZONTAL

SLIDER
736
34
928
67
schema-share-effectiveness
schema-share-effectiveness
0
100.0
85.0
5
1
%
HORIZONTAL

BUTTON
54
490
246
538
load imported network (if used)
load-network
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
264
491
362
538
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
737
180
1125
470
Network Status
time
% of nodes
0.0
200.0
0.0
110.0
true
true
"" ""
PENS
"susceptible" 1.0 0 -13345367 true "" "plot (count turtles with [susceptible?]) / (count turtles) * 100"
"updated" 1.0 0 -2674135 true "" "plot (count turtles with [updated?]) / (count turtles) * 100"
"agreed" 1.0 0 -13840069 true "" "plot (count turtles with [agreed?]) / (count turtles) * 100"
"implemented" 1.0 0 -8431303 true "" "plot (count turtles with [implemented?]) / (count turtles) * 100"
"reviewed" 1.0 0 -2064490 true "" "plot (count turtles with [reviewed?]) / (count turtles) * 100"
"adjusted" 1.0 0 -817084 true "" "plot (count turtles with [adjusted?]) / (count turtles) * 100"
"resistant" 1.0 0 -7500403 true "" "plot (count turtles with [resistant?]) / (count turtles) * 100"

SLIDER
53
319
249
352
schema-check-frequency
schema-check-frequency
1
20
1.0
1
1
ticks
HORIZONTAL

SLIDER
736
104
928
137
problem-space-complexity
problem-space-complexity
0
100
90.0
5
1
NIL
HORIZONTAL

SLIDER
736
69
928
102
problem-solving-effectiveness
problem-solving-effectiveness
0
problem-space-complexity
60.0
1
1
NIL
HORIZONTAL

SLIDER
736
139
928
172
coevolution-constraints
coevolution-constraints
0
100
65.0
5
1
NIL
HORIZONTAL

BUTTON
363
491
464
538
go-2
go-2
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
1359
160
1477
205
resistant-node-count
count turtles with [resistant?]
1
1
11

BUTTON
153
396
246
442
2. layout
layout
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
54
443
151
489
3. trigger
trigger
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
54
261
249
294
initial-detection-size
initial-detection-size
0
100
14.0
1
1
NIL
HORIZONTAL

CHOOSER
55
125
250
170
links-to-use
links-to-use
"directed" "undirected"
0

BUTTON
1139
337
1341
370
betweenness
betweenness
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1139
402
1341
435
closeness
closeness
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1139
370
1341
403
eigenvector
eigenvector
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1138
212
1240
257
links
count links
17
1
11

MONITOR
1240
212
1342
257
nodes
count turtles
17
1
11

BUTTON
54
396
151
442
1. setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1138
256
1342
301
mean path length
nw:mean-path-length
5
1
11

BUTTON
1139
434
1341
470
community detection
community-detection
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
1137
34
1342
204
degree distribution
degree
Frequency
0.0
15.0
0.0
350.0
false
false
"set-histogram-num-bars 6" ""
PENS
"default" 1.0 1 -16777216 true "" "histogram [ count my-links ] of turtles"

TEXTBOX
112
10
266
36
Setup Network
12
0.0
1

TEXTBOX
1451
10
1533
28
Outcomes
12
0.0
1

TEXTBOX
1183
10
1333
28
Network Features
12
0.0
1

TEXTBOX
856
10
1006
28
CQIN Composite Factors
12
0.0
1

TEXTBOX
1187
315
1337
333
Centrality Measures
12
0.0
1

SLIDER
934
34
1125
67
learning-gain
learning-gain
1
2
1.1
0.05
1
NIL
HORIZONTAL

INPUTBOX
1358
35
1476
95
signal-base
0.0
1
0
Number

INPUTBOX
1478
35
1597
95
signal-increment
10.0
1
0
Number

INPUTBOX
1358
97
1476
157
signal-target
100.0
1
0
Number

PLOT
1358
270
1603
470
outcome signal
Time units
Signal level
0.0
100.0
90.0
100.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Signal"

SLIDER
55
191
249
224
number-of-nodes
number-of-nodes
2
500
94.0
1
1
NIL
HORIZONTAL

SLIDER
55
226
249
259
node-degree
node-degree
1
10
2.9
0.1
1
NIL
HORIZONTAL

CHOOSER
55
79
250
124
network-type
network-type
"random" "preferential-attachment"
1

CHOOSER
55
34
250
79
network-source
network-source
"Import" "Generate"
1

BUTTON
464
491
565
538
go-3
go-3
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
567
491
667
538
go-4
go-4
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
668
491
769
538
go-5
go-5
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
52
354
249
387
planned-iterations
planned-iterations
0
100
20.0
1
1
NIL
HORIZONTAL

MONITOR
1480
159
1599
204
Iteration-count
iteration-count
17
1
11

BUTTON
771
491
865
538
cycle
cycle
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
1479
97
1598
158
max-non-co-operating
0.5
1
0
Number

OUTPUT
1358
211
1602
264
11

@#$#@#$#@
## THE COMPLEX QUALITY IMPROVEMENT NETWORK (CQIN) AGENT-BASED MODEL

This model explores the behaviour of quality improvement (QI) teams seeking improvement within a complex adaptive system. The starting point is the agent communication and cooperation underpinning the activity. Any improvement (or lack of improvement) is an emergent property of the system arising from this granular cooperative activity.

The CQIN agent-based model treats large-scale improvement activity as a network of adaptive agents cooperating to solve a problem. To cooperate effectively, agents require a shared understanding of the improvement goals, the current state of their actions, and how to progress towards the goal state. The working premise is that maintaining a shared representation of the problems, solution options, problem-solving effectiveness and progress towards goals are prerequisites to the necessary cooperation. These factors drive the probability of success and how quickly the goal state can be achieved within the unique constraints of the problem, the networked participants, and external constraints.

## HOW THE CQIN MODEL WORKS

Model function is based on the idea of the spread of a virus through a computer network. In our case, it is information that is shared amongst linked neighbours. This information may then be accepted or rejected. At agent level, the behaviour is 'share', 'update'  or 'reject' the information. At the system level, the spreading process, once triggered, cycles through five quality improvement phases analogous to a plan-do-study-act cycle, with the cycles proceeding through a planned number of iterations. Each completed iteration incrementally updates progress towards the 'target state'. 'Information' as used here is an abstract notion - it is whatever the relevant message is in relation to the quality improvement phase and may include acting on the information in a coordinated manner (e.g. "we need to cooperate to do this together"). Stop conditions can be set for reaching the target, maximum number of iterations, or falling below a minimum level of network cooperation.

The process commences with detection of an environmental signal of interest by a pre-determined number of the agents. Each agent with this new information then seeks to share the update with their network neighbours. Sharing follows a transmit and receive format, where the transmission must first succeed, and then also be received (or accepted). Each failed share has a chance of rejection and making the receiving agent uncooperative. For each agent interaction to be successful, a probability threshold must be exceeded. The thresholds are determined by the interaction between the variables that apply to each QI phase. A high value for the product of variable interactions results in a correspondingly high chance of sharing/cooperation; conversely, a low value for the product of the variable interaction increases the chances of rejection and reducing network cooperation.

## HOW TO USE THIS MODEL

* Setup
* If generating a network, set the number of nodes, network connection type, and node degree
* If loading a network, use the "Load Network" button (refer code comments for more details)
* Generate or import the network
* Layout the network
* Set desired values for the CQIN composite factors (Initial detection, planned iterations,  schema share effectiveness, problem solving capability, coevolution constraints, problem space complexity, learning gain, rejection threshold, rejection chance)
* Set the base improvement signal level, increments and target level
* Set maximum level of network dropout (non-cooperation)
* Trigger
* To complete one iteration, run Go, Go-1, Go-2, Go-3 Go-4, Go-5.Each phase will stop when all agents are updated or resistant, or a stop condition is reached.
* To run the next iteration, use 'Cycle', then repeat the 'Go' sequence
* Use behaviour space to run as an automated sequence


## QUESTIONS AND ONGOING WORK

* The threshold formulas are starting points based on the hypothesised relationships between the CQIN factors, quality improvement theory and literature (please refer to the research that accompanies this model). Whilst plausible and practical, they are almost certainly insufficiently defined and ongoing verification with real-world examples will  refine them.

* This version allows resistant actors to rejoin the network each new iteration. This is premised on each new iteration (if done well) being an opportunity to reset and try again, this time with the learning gain from the previous iteration. It is assumed there will be real-world examples where this reset may not be able to applicable.


## HOW TO CITE

If you mention this model or the NetLogo software in a publication, please include the citations below.

* William Wilson (2020-2023) School of Food and Advanced Technology, Massey University, New Zealand  

Related rersearch:
William Wilson, Scott McLachlan, Kudakwashe Dube, Kathleen Potter & Nihal Jayamaha (2023) Uncertainty, emergence and adaptation: A complex adaptive systems approach to quality improvement, Quality Management Journal, 30:3, 168-186, DOI: 10.1080/10686967.2023.2211287


This model was inspired and adapted from:

* Stonedahl, F. and Wilensky, U. (2008).  NetLogo Virus on a Network model.  http://ccl.northwestern.edu/netlogo/models/VirusonaNetwork.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

The original code for the two underlying ideas in the Stonedahl and Wilensky model:

;; to spread-virus
;;  ask turtles with [infected?]
;;    [ ask link-neighbors with [not resistant?]
;;       [ if random-float 100 < virus-spread-chance
;;            [ become-infected ] ] ]
;; end

;; to do-virus-checks
;;  ask turtles with [infected? and virus-check-timer = 0]
;;  [
;;    if random 100 < recovery-chance
;;    [
;;      ifelse random 100 < gain-resistance-chance
;;        [ become-resistant ]
;;        [ become-susceptible ]
;;    ]
;;  ]
;; end

The major feature of the original retained is the idea of the schema-check function as the basis for the agents to determine their state. The CQIN model expands the number of agent states, the spreading activity via phases and iterations, input variables, creates new threshold formulas, stop conditions, output reporting and network measures.

Code for the centrality measures and the ability to import an external network graph is sourced from the Netlogo model library - 'Extensions NW General' Examples

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2008 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

<!-- 2008 Cite: Stonedahl, F. -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Case 3 v1" repetitions="1" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="signal-share-chance">
      <value value="90"/>
      <value value="70"/>
      <value value="50"/>
      <value value="30"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="410"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="psc-weight">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gain-resistance-chance">
      <value value="5"/>
      <value value="10"/>
      <value value="30"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ssc-weight">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-check-frequency">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="-5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="9"/>
      <value value="7"/>
      <value value="5"/>
      <value value="3"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-proportion-sharing">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cc-weight">
      <value value="48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pse-weight">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="2"/>
      <value value="2.5"/>
      <value value="3"/>
      <value value="3.5"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="links-to-use">
      <value value="&quot;directed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="14"/>
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-effectiveness">
      <value value="90"/>
      <value value="70"/>
      <value value="50"/>
      <value value="30"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="80"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Deprescribing Project Run1" repetitions="1" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="signal-share-chance">
      <value value="90"/>
      <value value="70"/>
      <value value="50"/>
      <value value="30"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="410"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gain-resistance-chance">
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-check-frequency">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="-0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="10"/>
      <value value="5"/>
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-proportion-sharing">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.05"/>
      <value value="1.1"/>
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="links-to-use">
      <value value="&quot;directed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="28"/>
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-effectiveness">
      <value value="90"/>
      <value value="70"/>
      <value value="50"/>
      <value value="30"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="80"/>
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="80"/>
      <value value="70"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Deprescribing Project Estimate 18 August" repetitions="200" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="schema-share-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="410"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="-0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-threshold">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-capability">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-non-co-operating">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="COVID-19 Policy Response Iterations" repetitions="200" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="rejection-threshold">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-check-frequency">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-share-effectiveness">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-proportion-sharing">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="human-communication-factors">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-effectiveness">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="80"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Deprescribing_25 August Runs" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="rejection-threshold">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-non-co-operating">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="410"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-check-frequency">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="-0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-share-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="links-to-use">
      <value value="&quot;directed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="90"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demo" repetitions="100" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="rejection-threshold">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-non-co-operating">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="410"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-check-frequency">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="-0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-share-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="links-to-use">
      <value value="&quot;directed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="90"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Survey variations" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup
layout
trigger</setup>
    <go>go
if all? turtles [updated? or agreed? or resistant?]
[go-2]
if all? turtles [agreed?  or implemented? or resistant?]
[go-3]
if all? turtles [implemented? or reviewed? or resistant?]
[go-4]
if all? turtles [reviewed? or adjusted? or resistant?]
[go-5]
if all? turtles [adjusted? or resistant?]
[cycle]</go>
    <metric>ticks</metric>
    <metric>count turtles with [resistant?]</metric>
    <metric>signal</metric>
    <metric>iteration-count</metric>
    <enumeratedValueSet variable="rejection-threshold">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-non-co-operating">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="410"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-check-frequency">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-source">
      <value value="&quot;Generate&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-increment">
      <value value="-0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rejection-chance">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="schema-share-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="planned-iterations">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-solving-effectiveness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="node-degree">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-gain">
      <value value="1.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-detection-size">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-base">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="links-to-use">
      <value value="&quot;directed&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-space-complexity">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="signal-target">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="coevolution-constraints">
      <value value="80"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
