;--------------------------------------------------------------------------------------;
;    This model is free software: you can redistribute it and/or modify                ;
;    it under the terms of the GNU General Public License as published by the          ;
;    Free Software Foundation, either version 3 of the License, or (at your option)    ;
;    any later version.                                                                ;
;                                                                                      ;
;    This program is distributed in the hope that it will be useful, but               ;
;    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY        ;
;    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License           ;
;    for more details.                                                                 ;
;                                                                                      ;
;    For the full GNU General Public License, please see                               ;
;    <https://www.gnu.org/licenses/>.                                                  ;
;                                                                                      ;
;    Author: Natalie Davis                                                             ;
;--------------------------------------------------------------------------------------;

extensions [ table rnd csv ]

breed [ consumers consumer ]
breed [ diets diet ]
undirected-link-breed [ consumer-diets consumer-diet ]
undirected-link-breed [ social-contacts social-contact ]

globals [
  this-seed
]

consumers-own [
  cost-motivation
  taste-motivation
  ethics-motivation
  health-motivation
  social-information-adherence
  contacts-this-timestep
  household-income
  max-willingness-to-spend
  household-size
  satisfaction-threshold
  current-diet
  current-diet-utility-score
  max-diet-utility-score
  min-diet-utility-score
  perceptions-set? ; for set up only
  household-set? ; for set up only
]

diets-own [
  name
  cost
]

consumer-diets-own [
  diet-name
  financially-feasible?
  perceived-cost
  perceived-taste
  perceived-ethics
  perceived-health
  times-consumed
]

social-contacts-own [
  is-household-member?
  is-close-friend?
  link-strength
  similarity
  ; trust
  no-of-interactions
  interacted-this-timestep?
]

to setup
  clear-all
  reset-ticks
  ; Set up seed for writing, re-creating output if needed
  ifelse my-seed != 0 and round my-seed = my-seed [ set this-seed my-seed ] [ set this-seed new-seed ]
  random-seed this-seed
  make-consumers
  make-diets
  make-consumer-diet-links
  make-social-network
  record-consumer-baseline-data
  record-run-params
end

to go
  ask social-contacts [
    set interacted-this-timestep? false
  ]
  ask consumers [
    set contacts-this-timestep 0
    update-diet-utility
    choose-diet
    if social-interaction? [
      discuss-with-social-network
    ]
    update-taste-perception
  ]
  ; Record data for this timestep
  record-consumer-data
  record-network-data
  if network-structural-change? [
    ; Update network structure for next timestep
    update-links
  ]
  tick
end

to choose-diet
  ; Choose new if normalized utility of current diet is below satisfaction threshold or if max. utility score is 0 (i.e. at initialization)
  let choose-new? ifelse-value (max-diet-utility-score = 0 or current-diet-utility-score < satisfaction-threshold)
  [ true ][ false ]
  if choose-new? [
    ; Evaluate each diet and assign score
    let diet-scores table:make
    let diet-score-sum 0
    let max-utility-score max-diet-utility-score
    ; If min. diet utility score = 0 (e.g. at initialization), set min. utility score temp variable here to 100 to force update
    let min-utility-score ifelse-value min-diet-utility-score > 0 [ min-diet-utility-score ] [ 100 ]
    ask my-consumer-diets [
      ; diet score - product of constraints (0/1) and motivations * perceptions, normalized to [0, 1] for weighting in selection (with infeasible diets automatically = 0)
      let diet-score ifelse-value include-health-motivation? [
        ifelse-value include-ethics-motivation? [
          financially-feasible? * (((
            [ cost-motivation ] of myself * perceived-cost +
            [ taste-motivation ] of myself * perceived-taste +
            [ ethics-motivation ] of myself * perceived-ethics +
            [ health-motivation ] of myself * perceived-health
          ) + 1) / 2)
        ] [
          financially-feasible? * (((
            [ cost-motivation ] of myself * perceived-cost +
            [ taste-motivation ] of myself * perceived-taste +
            [ health-motivation ] of myself * perceived-health
          ) + 1) / 2)
        ]
      ] [
        financially-feasible? * (((
          [ cost-motivation ] of myself * perceived-cost +
          [ taste-motivation ] of myself * perceived-taste
        ) + 1) / 2)
      ]
      table:put diet-scores diet-name diet-score
      ; Update min, max, total utilities
      if diet-score > max-utility-score [ set max-utility-score diet-score ]
      if diet-score < min-utility-score [ set min-utility-score diet-score ]
      set diet-score-sum diet-score-sum + diet-score
    ]
    ; Update consumer's min and max diet utility scores from this round of calculating utilities
    set max-diet-utility-score max-utility-score
    set min-diet-utility-score min-utility-score
    ; Check if no diets scored >= 0 (i.e. none are suitable)
    ifelse (max table:values diet-scores = 0) [
      ; If no diets are suitable, choose least expensive diet
      set current-diet one-of diets with-min [ cost ]
      set current-diet-utility-score 0
    ] [
      ; If all diets scored equally (unlikely), choose least expensive diet
      ifelse max-diet-utility-score - min-diet-utility-score = 0 [
        set current-diet one-of diets with-min [ cost ]
        set current-diet-utility-score max-diet-utility-score
      ] [
        ; Choose diet using 'roulette wheel' selection with score as weight (probability of being selected)
        let selection rnd:weighted-one-of-list table:to-list diet-scores [ [ a ] -> last a / diet-score-sum ]
        ; Set current diet of consumer to diet named in selection, save utility score for selected diet
        set current-diet one-of diets with [ name = first selection ]
        set current-diet-utility-score last selection
      ]
    ]
  ]
  ask my-consumer-diets with [ diet-name = [ name ] of [ current-diet ] of myself ] [ set times-consumed times-consumed + 1 ]
end

to discuss-with-social-network
  ; Sort possible social contacts by link strength, then similarity
  let sorted-social-contacts sort-by [ [ x y ] ->
    ([ link-strength ] of x > [ link-strength ] of y) or
    ([ link-strength ] of x >= [ link-strength ] of y and [ similarity ] of x > [ similarity ] of y) or
    ([ link-strength ] of x = [ link-strength ] of y and [ similarity ] of x > [ similarity ] of y)
    ; only include other social contacts who have not used up all of their social interactions for timestep
  ] my-social-contacts with [ [ contacts-this-timestep ] of (ifelse-value end1 = myself [ end2 ] [ end1 ]) < max-contacts-per-timestep ]
  ; Each consumer tests if they will contact each of the other agents on their list
  while [ contacts-this-timestep < max-contacts-per-timestep and length sorted-social-contacts > 0 ] [
    let next-contact first sorted-social-contacts
    let alter ifelse-value [ end1 ] of next-contact = self [ [ end2 ] of next-contact ] [ [ end1 ] of next-contact ]
    ifelse sum [ link-strength ] of my-social-contacts = 0 or (sum [ link-strength ] of my-social-contacts - [ link-strength ] of next-contact) = 0 [
      ; If I don't have anyone (else) in particular I would interact with...
      if ([ similarity ] of next-contact > random-float 1) [
        exchange-influence alter
        set contacts-this-timestep contacts-this-timestep + 1
        ask next-contact [ set no-of-interactions no-of-interactions + 1 ]
      ]
    ] [
      if (
        ; or we're (1) connected already
        ([ link-strength ] of next-contact > 0 and ([ link-strength ] of next-contact / (sum [ link-strength ] of my-social-contacts - [ link-strength ] of next-contact) > random-float 1)) or
        ; or (2) completely unknown to each other and our friends, but similar enough that we might cross paths...
        ((([ similarity ] of next-contact / sum [ link-strength ] of my-social-contacts) > random-float 1)) ) [
        ; then we exchange influence
        exchange-influence alter
        set contacts-this-timestep contacts-this-timestep + 1
        ask next-contact [
          set no-of-interactions no-of-interactions + 1
          set interacted-this-timestep? true
        ]
      ]
    ]
    set sorted-social-contacts remove next-contact sorted-social-contacts
  ]

  ; Default interaction with household members
  ask my-social-contacts with [ is-household-member? ] [
    let hh-member ifelse-value end1 = myself [ end2 ] [ end1 ]
    ask myself [ exchange-influence hh-member ]
    set no-of-interactions no-of-interactions + 1
  ]
  set contacts-this-timestep contacts-this-timestep + count social-contacts with [ is-household-member? ]
end

to exchange-influence [ friend ]
  ; Store perceptions as temporary variables to allow for synchronous updating
  let current-diet-this-friend [ current-diet ] of friend
  let current-diet-myself current-diet
  let friend-perceptions-my-diet-ethics [ perceived-ethics ] of consumer-diet [ who ] of friend [ who ] of current-diet
  let friend-perceptions-my-diet-health [ perceived-health ] of consumer-diet [ who ] of friend [ who ] of current-diet
  let my-perceptions-friend-diet-ethics [ perceived-ethics ] of consumer-diet who [ who ] of current-diet-this-friend
  let my-perceptions-friend-diet-health [ perceived-health ] of consumer-diet who [ who ] of current-diet-this-friend
  let l [ link-strength ] of social-contact who [ who ] of friend

  ; Influence from alter to ego regarding alter's diet
  ask my-consumer-diets with [ diet-name = [ name ] of current-diet-this-friend ] [
    ifelse ((l + abs([ ethics-motivation ] of friend - [ ethics-motivation ] of myself)) / 2) > random-float 1 [ ; influence accepted
      set my-perceptions-friend-diet-ethics max list -1 min list 1 my-perceptions-friend-diet-ethics + [ social-information-adherence ] of myself *
      ([ perceived-ethics ] of consumer-diet [ who ] of friend [ who ] of current-diet-this-friend - my-perceptions-friend-diet-ethics)
    ][
      if ((2 - l - abs([ ethics-motivation ] of friend - [ ethics-motivation ] of myself)) / 2) > random-float 1 [ ; influence rejected
        set my-perceptions-friend-diet-ethics max list -1 min list 1 my-perceptions-friend-diet-ethics - [ social-information-adherence ] of myself *
        ([ perceived-ethics ] of consumer-diet [ who ] of friend [ who ] of current-diet-this-friend - my-perceptions-friend-diet-ethics)
      ] ; else influence ignored
    ]

    ifelse ((l + abs([ health-motivation ] of friend - [ health-motivation ] of myself)) / 2) > random-float 1 [
      set my-perceptions-friend-diet-health max list -1 min list 1 my-perceptions-friend-diet-health + [social-information-adherence ] of myself *
      ([ perceived-health ] of consumer-diet [ who ] of friend [ who ] of current-diet-this-friend - my-perceptions-friend-diet-health)
    ][
      if ((2 - l - abs([ health-motivation ] of friend - [ health-motivation ] of myself)) / 2) > random-float 1 [
        set my-perceptions-friend-diet-health max list -1 min list 1 my-perceptions-friend-diet-health - [ social-information-adherence ] of myself *
        ([ perceived-health ] of consumer-diet [ who ] of friend [ who ] of current-diet-this-friend - my-perceptions-friend-diet-health)
      ]
    ]
  ]

  ; Influence from ego to alter regarding ego's diet
  ask [ my-consumer-diets with [ diet-name = [ name ] of current-diet-myself ] ] of friend [
    ifelse ((l + abs([ ethics-motivation ] of myself - [ ethics-motivation ] of friend)) / 2) > random-float 1 [
      set friend-perceptions-my-diet-ethics max list -1 min list 1 friend-perceptions-my-diet-ethics + [ social-information-adherence ] of friend *
      ([ perceived-ethics ] of consumer-diet [ who ] of myself [ who ] of current-diet-this-friend - friend-perceptions-my-diet-ethics)
    ][
      if ((2 - l - abs([ ethics-motivation ] of myself - [ ethics-motivation ] of friend)) / 2) > random-float 1 [
        set friend-perceptions-my-diet-ethics max list -1 min list 1 friend-perceptions-my-diet-ethics - [ social-information-adherence ] of friend *
        ([ perceived-ethics ] of consumer-diet [ who ] of myself [ who ] of current-diet-this-friend - friend-perceptions-my-diet-ethics)
      ]
    ]

    ifelse ((l + abs([ health-motivation ] of myself - [ health-motivation ] of friend)) / 2) > random-float 1 [
      set friend-perceptions-my-diet-health max list -1 min list 1 friend-perceptions-my-diet-health + [ social-information-adherence ] of friend *
      ([ perceived-health ] of consumer-diet [ who ] of myself [ who ] of current-diet-this-friend - friend-perceptions-my-diet-health)
    ][
      if ((2 - l - abs([ health-motivation ] of myself - [ health-motivation ] of friend)) / 2) > random-float 1 [
        set friend-perceptions-my-diet-health max list -1 min list 1 friend-perceptions-my-diet-health - [ social-information-adherence ] of friend *
        ([ perceived-health ] of consumer-diet [ who ] of myself [ who ] of current-diet-this-friend - friend-perceptions-my-diet-health)
      ]
    ]
  ]

  ; Update actual perceptions from temp. variables
  ask my-consumer-diets with [ diet-name = [ name ] of current-diet-this-friend ] [
    set perceived-ethics my-perceptions-friend-diet-ethics
    set perceived-health my-perceptions-friend-diet-health
  ]
  ask [ my-consumer-diets with [ diet-name = [ name ] of current-diet-myself ] ] of friend [
    set perceived-ethics friend-perceptions-my-diet-ethics
    set perceived-health friend-perceptions-my-diet-health
  ]

  ; Update no. of contacts
  ask friend [
    set contacts-this-timestep contacts-this-timestep + 1
  ]
end

to update-taste-perception
  ask my-consumer-diets [
    ifelse diet-name = ([ [ name ] of current-diet ] of myself) [
      set perceived-taste perceived-taste + ((1 - perceived-taste) / (1 + taste-preference-change-gradient * exp
        ((taste-preference-change-error * random-binomial) + ((- taste-preference-change-rate) * times-consumed))))
    ] [
      set perceived-taste (- 1) + ((perceived-taste + 1) / (1 + (1 / taste-preference-change-gradient) * exp
        (((- taste-preference-change-rate) * (ticks - times-consumed)))))
    ]
  ]
end

to update-links
  ask social-contacts with [ not is-household-member? and not is-close-friend? ] [
    ifelse interacted-this-timestep? [
      set link-strength link-strength + ((1 - link-strength) / (1 + link-strength-change-gradient * exp
        ((- link-strength-change-rate) * no-of-interactions)))
    ] [
      set link-strength (link-strength / (1 + (1 / link-strength-change-gradient) * exp
        ((- link-strength-change-rate) * (ticks - no-of-interactions))))
    ]
  ]
end

to update-diet-utility
  let my-current-consumer-diet my-consumer-diets with [ diet-name = [ name ] of [ current-diet ] of myself ]
  let updated-diet-utility-score 0
  ask my-current-consumer-diet [
    set updated-diet-utility-score ifelse-value include-health-motivation? [
      ifelse-value include-ethics-motivation? [
        financially-feasible? * (((
          [ cost-motivation ] of myself * perceived-cost +
          [ taste-motivation ] of myself * perceived-taste +
          [ ethics-motivation ] of myself * perceived-ethics +
          [ health-motivation ] of myself * perceived-health
        ) + 1) / 2)
      ] [
        financially-feasible? * (((
          [ cost-motivation ] of myself * perceived-cost +
          [ taste-motivation ] of myself * perceived-taste +
          [ health-motivation ] of myself * perceived-health
        ) + 1) / 2)
      ]
    ] [
      financially-feasible? * (((
        [ cost-motivation ] of myself * perceived-cost +
        [ taste-motivation ] of myself * perceived-taste
      ) + 1) / 2)
    ]
  ]
  set current-diet-utility-score updated-diet-utility-score
end

;---------------------------------------------------;
; Setup methods                                     ;
;---------------------------------------------------;
to make-consumers
  create-consumers n-consumers [
    set cost-motivation 0
    set taste-motivation 0
    set ethics-motivation 0
    set health-motivation 0
    set social-information-adherence 0
    set contacts-this-timestep 0
    set household-income 0
    set max-willingness-to-spend max list 0 min list 1 random-normal mean-max-willingness-to-spend sd-max-willingness-to-spend
    set household-size 0
    set satisfaction-threshold max list 0 min list 1 random-normal mean-satisfaction-threshold sd-satisfaction-threshold
    set current-diet-utility-score 0
    set max-diet-utility-score 0
    set min-diet-utility-score 0
    set shape "person"
    set perceptions-set? false
    set household-set? false
  ]

  ;; Set up population demographics (household size, income) from input file
  ; Household size
  file-open household-file
  let headers file-read-line ; Skip header row
  let n-adults 0
  let this-household-size 0
  while [ not file-at-end? ] [
    let household-dat csv:from-row file-read-line ; Format should be frequency, no. of adults, no. of children, weekly income lower bound, weekly income upper bound
    set n-adults item 1 household-dat
    set this-household-size item 1 household-dat + item 2 household-dat
    let this-consumer-set up-to-n-of round ((first household-dat * n-consumers) / n-adults) consumers with [ household-set? = false ]
    ask consumers with [ member? self this-consumer-set ] [
      ; Check that consumer wasn't added to household of another consumer in this-consumer-set
      if household-set? = false [
        set household-size this-household-size ; no. of adults + no. of children
        set household-income random-float item 3 household-dat + random-float (item 4 household-dat - item 3 household-dat)
        ; If there is >1 adult in the household, group adults together and create household-member social ties
        if n-adults > 1 [
          let i 1
          while [ i < n-adults ] [
            let new-hh-member nobody
            ifelse any? other consumers with [ member? self this-consumer-set and not household-set? ] [
              set new-hh-member one-of other consumers with [ member? self this-consumer-set and not household-set? ]
            ] [
              set new-hh-member one-of other consumers with [ household-size = 0 or household-size = 1 ]
            ]
            create-social-contact-with new-hh-member [
              set is-household-member? true
              set is-close-friend? false
              set link-strength max list 0 min list 1 random-normal mean-household-link-strength sd-household-link-strength
              set similarity 1
              set no-of-interactions 0
            ]
            ask new-hh-member [
              set household-income [ household-income ] of myself
              set household-size [ household-size ] of myself
              set household-set? true
            ]
            set i i + 1
          ]
        ]
        set household-set? true
      ]
    ]

    ; If anyone didn't get the right number of adult household members, group them up with one/some of the single people
    if n-adults > 1 and any? consumers with [ household-size = this-household-size and count my-social-contacts with [ is-household-member? ] = 0 ] [
      ask consumers with [ household-size = this-household-size and count my-social-contacts with [ is-household-member? ] = 0 ] [
        let i 1
        let new-hh-member nobody
        while [ i < n-adults ] [
          ifelse any? other consumers with [ household-size = this-household-size and count my-social-contacts with [ is-household-member? ] < n-adults ] [
            set new-hh-member one-of other consumers with [ household-size = this-household-size and count my-social-contacts with [ is-household-member? ] < n-adults ]
          ] [
            set new-hh-member one-of other consumers with [ household-size = 1 ]
          ]
          create-social-contact-with new-hh-member [
            set is-household-member? true
            set is-close-friend? false
            set link-strength max list 0 min list 1 random-normal mean-household-link-strength sd-household-link-strength
            set similarity 1
            set no-of-interactions 0
          ]
          ask new-hh-member [
            set household-size [ household-size ] of myself
            set household-income [ household-income ] of myself
          ]
          set i i + 1
        ]
      ]
    ]
  ]
  file-close

  ; Handle any consumers with missing household size or income data (shouldn't happen) - set as single-person household with random income bracket of single-person households
  ask consumers with [ household-size = 0 or household-income = 0 ] [
    set household-size 1
    set household-income max list 1 random-float max [ household-income ] of consumers with [ household-size = 1 ]
  ]

  ; Arrange consumers in a large cirle, making sure that household members are not next to each other (for setting up non-household social network later)
  layout-circle (sort-by [ [ a b ] -> not member? b [ my-social-contacts ] of a ] consumers ) max-pxcor - 1

  ;; Set up social information adherence
  file-open social-information-adherence-file
  set headers file-read-line ; Skip header row
  let majority 0
  let majority-dat []
  while [ not file-at-end? ] [
    let adherence-dat csv:from-row file-read-line ; Format should be frequency, mean, SD
    if (first adherence-dat > majority) [ set majority-dat adherence-dat]
    ask n-of (first adherence-dat * n-consumers) consumers with [ social-information-adherence = 0 ] [
      set social-information-adherence (random-normal item 1 adherence-dat item 2 adherence-dat) * soc-info-adherence-scaling-factor
    ]
  ]
  file-close
  ; Assign any remaining agents (there should be very few, if any) to majority norm adherence bracket
  ask consumers with [ social-information-adherence = 0 ] [
    set social-information-adherence (random-normal item 1 majority-dat item 2 majority-dat) * soc-info-adherence-scaling-factor
  ]
end

to make-diets
  create-diets 1 [
    set name "vegan"
;    set color green
    hide-turtle
  ]
  create-diets 1 [
    set name "vegetarian"
;    set color cyan
    hide-turtle
  ]
  create-diets 1 [
    set name "pescatarian"
;    set color blue
    hide-turtle
  ]
  create-diets 1 [
    set name "flexitarian"
;    set color yellow
    hide-turtle
  ]
  create-diets 1 [
    set name "omnivore"
;    set color red
    hide-turtle
  ]
  file-open diet-cost-file
  let headers file-read-line ; Format should be diet name, cost per week
  while [ not file-at-end? ] [
    let diet-cost-data csv:from-row file-read-line
    ask one-of diets with [ name = first diet-cost-data ] [
      set cost item 1 diet-cost-data
    ]
  ]
  file-close
  layout-circle diets (max-pxcor / 2.5)
 end

to make-consumer-diet-links
  let min-diet-cost one-of [ cost ] of diets with-min [ cost ]
  let max-diet-cost one-of [ cost ] of diets with-max [ cost ]
  ask consumers [
   create-consumer-diets-with diets [
      let consumer-end ifelse-value (is-consumer? end1) [ end1 ] [ end2 ]
      let diet-end ifelse-value (is-diet? end2) [ end2 ] [ end1 ]
      let is [ household-income ] of consumer-end * [ max-willingness-to-spend ] of consumer-end
      let h [ household-size ] of consumer-end
      let c [ cost ] of diet-end
      let hc (h * c) - (h * c * ((h - 1) * 0.1)) ; scaled to incorporate discount for households >1
      set diet-name [ name ] of diet-end
      set financially-feasible? ifelse-value hc > is [ 0 ] [ 1 ]
      set perceived-cost ifelse-value (is = 0 or financially-feasible? = 0) [ -1 ] [ -2 * (hc / is) + 1 ]
      hide-link
    ]
  ]

  ; Read in perceptions/motivations data
  file-open motivation-file
  let headers file-read-line ; Skip header row
  let majority 0
  let majority-dat []
  while [ not file-at-end? ] [
    let percep-dat csv:from-row file-read-line ; Format should be frequency, then mean and SD for each cost, taste, ethics, health motivations, respectively, then
    ; mean taste, mean health, mean ethics perceptions for each omnivore, flexitarian, pescetarian, vegetarian, vegan
    if (first percep-dat > majority) [ set majority-dat percep-dat]
    ask n-of (first percep-dat * n-consumers) consumers with [ not perceptions-set? ] [
      set cost-motivation random-normal item 1 percep-dat item 2 percep-dat
      set taste-motivation random-normal item 3 percep-dat item 4 percep-dat
      set ethics-motivation random-normal item 5 percep-dat item 6 percep-dat
      set health-motivation random-normal item 7 percep-dat item 8 percep-dat
      ask my-consumer-diets with [ diet-name = "omnivore" ] [
        set perceived-taste random-normal-in-bounds item 9 percep-dat 0.15 -1 1
        set perceived-health random-normal-in-bounds item 10 percep-dat 0.15 -1 1
        set perceived-ethics random-normal-in-bounds item 11 percep-dat 0.15 -1 1
      ]
      ask my-consumer-diets with [ diet-name = "flexitarian" ] [
        set perceived-taste random-normal-in-bounds item 12 percep-dat 0.15 -1 1
        set perceived-health random-normal-in-bounds item 13 percep-dat 0.15 -1 1
        set perceived-ethics random-normal-in-bounds item 14 percep-dat 0.15 -1 1
      ]
      ask my-consumer-diets with [ diet-name = "pescatarian" ] [
        set perceived-taste random-normal-in-bounds item 15 percep-dat 0.15 -1 1
        set perceived-health random-normal-in-bounds item 16 percep-dat 0.15 -1 1
        set perceived-ethics random-normal-in-bounds item 17 percep-dat 0.15 -1 1
      ]
      ask my-consumer-diets with [ diet-name = "vegetarian" ] [
        set perceived-taste random-normal-in-bounds item 18 percep-dat 0.15 -1 1
        set perceived-health random-normal-in-bounds item 19 percep-dat 0.15 -1 1
        set perceived-ethics random-normal-in-bounds item 20 percep-dat 0.15 -1 1
      ]
      ask my-consumer-diets with [ diet-name = "vegan" ] [
        set perceived-taste random-normal-in-bounds item 21 percep-dat 0.15 -1 1
        set perceived-health random-normal-in-bounds item 22 percep-dat 0.15 -1 1
        set perceived-ethics random-normal-in-bounds item 23 percep-dat 0.15 -1 1
      ]
      set perceptions-set? true
    ]
  ]
  file-close
  ; Assign any remaining agents (there should be very few, if any) to majority motivation bracket
  ask consumers with [ not perceptions-set? ] [
    set cost-motivation random-normal item 1 majority-dat item 2 majority-dat
    set taste-motivation random-normal item 3 majority-dat item 4 majority-dat
    set ethics-motivation random-normal item 5 majority-dat item 6 majority-dat
    set health-motivation random-normal item 7 majority-dat item 8 majority-dat
    ask my-consumer-diets with [ diet-name = "omnivore" ] [
      set perceived-taste random-normal-in-bounds item 9 majority-dat 0.15 -1 1
      set perceived-health random-normal-in-bounds item 10 majority-dat 0.15 -1 1
      set perceived-ethics random-normal-in-bounds item 11 majority-dat 0.15 -1 1
    ]
    ask my-consumer-diets with [ diet-name = "flexitarian" ] [
      set perceived-taste random-normal-in-bounds item 12 majority-dat 0.15 -1 1
      set perceived-health random-normal-in-bounds item 13 majority-dat 0.15 -1 1
      set perceived-ethics random-normal-in-bounds item 14 majority-dat 0.15 -1 1
    ]
    ask my-consumer-diets with [ diet-name = "pescatarian" ] [
      set perceived-taste random-normal-in-bounds item 15 majority-dat 0.15 -1 1
      set perceived-health random-normal-in-bounds item 16 majority-dat 0.15 -1 1
      set perceived-ethics random-normal-in-bounds item 17 majority-dat 0.15 -1 1
    ]
    ask my-consumer-diets with [ diet-name = "vegetarian" ] [
      set perceived-taste random-normal-in-bounds item 18 majority-dat 0.15 -1 1
      set perceived-health random-normal-in-bounds item 19 majority-dat 0.15 -1 1
      set perceived-ethics random-normal-in-bounds item 20 majority-dat 0.15 -1 1
    ]
    ask my-consumer-diets with [ diet-name = "vegan" ] [
      set perceived-taste random-normal-in-bounds item 21 majority-dat 0.15 -1 1
      set perceived-health random-normal-in-bounds item 22 majority-dat 0.15 -1 1
      set perceived-ethics random-normal-in-bounds item 23 majority-dat 0.15 -1 1
    ]
    set perceptions-set? true
  ]


  ; Re-scale consumers' motivations so the total is [0,1]
  ask consumers [
    if not include-health-motivation? [
      set health-motivation 0
    ]
    if not include-ethics-motivation? [
      set ethics-motivation 0
    ]
    let motivation-sum sum (list cost-motivation taste-motivation ethics-motivation health-motivation)
    set cost-motivation cost-motivation / motivation-sum
    set taste-motivation taste-motivation / motivation-sum
    set ethics-motivation ethics-motivation / motivation-sum
    set health-motivation health-motivation / motivation-sum
  ]

  ask consumers [
    ; Choose initial diet using same probability-based multi-criteria method
    choose-diet
  ]
end

to make-social-network
  ; Iterate over turtles to create initial lattice of specified initial node degree (code adapted from 'Small Worlds' demo model in Models Library, Wilensky 2015)
  let n 0
  while [ n < count consumers ] [
    let i 1
    while [ i <= (initial-avg-degree / 2) ] [
      ask consumer n [
        ; The network layout command in the create-consumers sub should keep household members apart (and these are only social contacts set so far), but check here just to be sure
        if not member? consumer ((n + i) mod count consumers) my-social-contacts [
          create-social-contact-with consumer ((n + i) mod count consumers) [
            set is-household-member? false
            set is-close-friend? false
            set link-strength max list 0 min list 1 random-normal mean-acquaintances-link-strength sd-acquaintances-link-strength
            set similarity consumer-similarity end1 end2
            set no-of-interactions 0
            set interacted-this-timestep? false
          ]
        ]
      ]
      set i i + 1
    ]
    set n n + 1
  ]
  ; Rewire each link according to specified rewiring probability
  rewire-links
end

;---------------------------------------------------;
; Data collection                                   ;
;---------------------------------------------------;
to record-consumer-data
  let consumer-output-file (word "output/x_" this-seed "_consumerdata.csv")
  if ticks = 0 and file-exists? consumer-output-file [ file-delete consumer-output-file ]
  file-open consumer-output-file
  if ticks = 0 [
    file-print (word "ticks,consumer,current_diet,perceptions_about,financially_feasible,perceived_cost,perceived_taste,perceived_ethics,perceived_health")
  ]
  ask consumers [
    ask my-consumer-diets with [ diet-name = "omnivore" ] [
      file-print (word ticks "," [ who ] of myself "," [ [ name ] of current-diet ] of myself "," diet-name "," financially-feasible? "," perceived-cost "," perceived-taste ","
        perceived-ethics "," perceived-health)
    ]
    ask my-consumer-diets with [ diet-name = "flexitarian" ] [
      file-print (word ticks "," [ who ] of myself "," [ [ name ] of current-diet ] of myself "," diet-name "," financially-feasible? "," perceived-cost "," perceived-taste ","
        perceived-ethics "," perceived-health)
    ]
    ask my-consumer-diets with [ diet-name = "pescatarian" ] [
      file-print (word ticks "," [ who ] of myself "," [ [ name ] of current-diet ] of myself "," diet-name "," financially-feasible? "," perceived-cost "," perceived-taste ","
        perceived-ethics "," perceived-health)
    ]
    ask my-consumer-diets with [ diet-name = "vegetarian" ] [
      file-print (word ticks "," [ who ] of myself "," [ [ name ] of current-diet ] of myself "," diet-name "," financially-feasible? "," perceived-cost "," perceived-taste ","
        perceived-ethics "," perceived-health)
    ]
    ask my-consumer-diets with [ diet-name = "vegan" ] [
      file-print (word ticks "," [ who ] of myself "," [ [ name ] of current-diet ] of myself "," diet-name "," financially-feasible? "," perceived-cost "," perceived-taste ","
        perceived-ethics "," perceived-health)
    ]
  ]
  file-close
end

to record-consumer-baseline-data
  let consumer-baseline-output-file (word "x_" this-seed "_consumerbaselinedata.csv")
  if ticks = 0 and file-exists? consumer-baseline-output-file [ file-delete consumer-baseline-output-file ]
  file-open consumer-baseline-output-file
  if ticks = 0 [ file-print (word "who,cost_motivation,taste_motivation,ethics_motivation,health_motivation,social_information_adherence,household_income,"
    "max_willingness_to_spend,household_size") ]
  ask consumers [
    file-print (word who "," cost-motivation "," taste-motivation "," ethics-motivation "," health-motivation "," social-information-adherence "," household-income ","
      max-willingness-to-spend "," household-size)
  ]
  file-close
end

to record-network-data
  let network-output-file (word "x_" this-seed "_networkdata.csv")
  if ticks = 0 and file-exists? network-output-file [ file-delete network-output-file ]
  file-open network-output-file
  if ticks = 0 [ file-print "ticks,end1,end2,is_household_member,is_close_friend,link_strength,similarity,no_of_interactions" ]
  ask social-contacts with [ link-strength > 0 or no-of-interactions > 0 ] [
    file-print (word ticks "," [ who ] of end1 "," [ who ] of end2 "," is-household-member? "," is-close-friend? "," link-strength "," similarity "," no-of-interactions)
  ]
  file-close
end

to record-run-params
  file-open run-param-output-file
  file-print (word this-seed "," n-consumers "," rewiring-probability "," initial-avg-degree "," mean-close-friends "," max-contacts-per-timestep ","
    link-strength-change-rate "," link-strength-change-gradient "," soc-info-adherence-scaling-factor "," taste-preference-change-rate ","
    taste-preference-change-error "," taste-preference-change-gradient "," mean-max-willingness-to-spend "," sd-max-willingness-to-spend ","
    mean-household-link-strength "," sd-household-link-strength "," mean-friends-link-strength "," sd-friends-link-strength "," mean-acquaintances-link-strength ","
    sd-acquaintances-link-strength "," mean-satisfaction-threshold "," sd-satisfaction-threshold "," social-interaction? "," network-structural-change? ","
    include-health-motivation? "," include-ethics-motivation?)
  file-close
end

;---------------------------------------------------;
; Helper methods                                    ;
;---------------------------------------------------;
; Report a normally-distributed random float within specified boundaries
; Credit: Seth Tisue https://stackoverflow.com/a/20233830
to-report random-normal-in-bounds [mid dev mmin mmax]
  let result random-normal mid dev
  if result < mmin or result > mmax
    [ report random-normal-in-bounds mid dev mmin mmax ]
  report result
end

to-report random-binomial
  ifelse random-float 1 < 0.5 [ report -1 ][ report 1 ]
end

to-report consumer-similarity [a b]
  let hh-size-range max [ household-size ] of consumers - min [ household-size ] of consumers
  if hh-size-range = 0 [ error "Household size range mubst be >0" ]
  let income-range max [ household-income ] of consumers - min [ household-income ] of consumers
  if income-range = 0 [ error "Income range must be >0" ]
  report (1 - abs (([ household-income ] of a - [ household-income ] of b) / income-range)) *
            (1 - abs (([ household-size ] of a - [ household-size ] of b) / hh-size-range))
end

to rewire-links
  ask consumers [
    ; Don't rewire household links
    ask my-social-contacts with [ not is-household-member? ] [
      if (random-float 1) < rewiring-probability [
        let skip false
        let ego myself
        let alter ifelse-value end1 = ego [ end2 ] [ end1 ]
        ; Identify consumers who are not the first consumer (ego) or ego's current social-contacts (inc. household members) with >0 link strength
        let poss-social-contacts consumers with [ self != ego and ((not social-contact-neighbor? ego or [ link-strength ] of social-contact [ who ] of ego [ who ] of self = 0)) ]
        if (count poss-social-contacts > 0) [
          ; Calculate probability of rewiring to each alter - inverse Euclidean distance of attributes
          let rewire-prob table:make
          ask poss-social-contacts [
            let prob consumer-similarity ego self
            table:put rewire-prob [ who ] of self prob
          ]
          ; Calculate probability of maintaining current social-contact connection
          let prob consumer-similarity alter ego
          table:put rewire-prob [ who ] of alter prob
          ; Choose new friend using 'roulette wheel' selection with probability as weight
          let new-social-contact consumer first (rnd:weighted-one-of-list table:to-list rewire-prob [ [ a ] -> last a ])
          ; If the new social-contact is the same as the old one, skip creating a new link (and don't kill the previous link)
          ifelse new-social-contact = alter [
            set skip true
          ] [
            ; Create new social contact
            ask ego [
              create-social-contact-with new-social-contact [
                set is-household-member? false
                set is-close-friend? false
                set link-strength max list 0 min list 1 random-normal mean-acquaintances-link-strength sd-acquaintances-link-strength
                set similarity consumer-similarity end1 end2
                set no-of-interactions 0
                set interacted-this-timestep? false
              ]
            ]
          ]
          if skip = false [
            ; Remove old link if a new one was created
            set link-strength 0
            set is-close-friend? false
            set is-household-member? false
            hide-link
          ]
        ]
      ]
    ]

  ]
  ; Create "close ties" social links
  ask social-contacts with [ not is-household-member? ] [
    if (mean-close-friends / initial-avg-degree) > random-float 1 [
      set is-close-friend? true
      set link-strength max list 0 min list 1 random-normal mean-friends-link-strength sd-friends-link-strength
    ]
  ]
  ; Create fully-connected network to store similarity and social distance, but set all other network links to link-strength = 0
  ask consumers [
    let ego self
    create-social-contacts-with other consumers with [ not member? self my-social-contacts ] [
      let alter ifelse-value end1 = ego [ end2 ] [ end1 ]
      set link-strength 0
      set is-close-friend? false
      set is-household-member? false
      set similarity consumer-similarity end1 end2
      set no-of-interactions 0
      hide-link
    ]
  ]
end

to-report avg-node-degree
  let all-node-degrees []
  ask consumers [ set all-node-degrees lput count my-social-contacts with [ not is-household-member? and link-strength > 0 ] all-node-degrees ]
  report mean all-node-degrees
end
@#$#@#$#@
GRAPHICS-WINDOW
205
6
633
435
-1
-1
12.73
1
10
1
1
1
0
0
0
1
-16
16
-16
16
1
1
1
ticks
30.0

SLIDER
20
90
195
123
n-consumers
n-consumers
10
1000
250.0
1
1
NIL
HORIZONTAL

BUTTON
35
45
98
78
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
120
45
183
78
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
20
125
195
158
rewiring-probability
rewiring-probability
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
20
160
195
193
initial-avg-degree
initial-avg-degree
0
n-consumers - 1
50.0
1
1
NIL
HORIZONTAL

INPUTBOX
655
15
875
75
my-seed
0.0
1
0
Number

BUTTON
75
10
147
43
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
885
255
1145
300
Avg. node degree (non-household members)
avg-node-degree
3
1
11

INPUTBOX
655
80
875
140
household-file
NIL
1
0
String

INPUTBOX
655
210
875
270
motivation-file
NIL
1
0
String

INPUTBOX
655
275
875
335
social-information-adherence-file
NIL
1
0
String

TEXTBOX
20
195
195
236
Network set up will only work as expected if initial avg. degree is an even number!
11
0.0
1

INPUTBOX
655
145
875
205
diet-cost-file
NIL
1
0
String

SLIDER
20
390
195
423
soc-info-adherence-scaling-factor
soc-info-adherence-scaling-factor
0
1
0.025
0.001
1
NIL
HORIZONTAL

INPUTBOX
655
340
875
400
run-param-output-file
NIL
1
0
String

SLIDER
20
285
195
318
max-contacts-per-timestep
max-contacts-per-timestep
0
100
10.0
1
1
NIL
HORIZONTAL

SLIDER
20
430
195
463
taste-preference-change-rate
taste-preference-change-rate
0
1
0.01
0.001
1
NIL
HORIZONTAL

PLOT
885
15
1145
250
Diet frequency
Ticks
No. of consumers
0.0
10.0
0.0
75.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot count consumers with [ [ name ] of current-diet = \"omnivore\" ] "
"pen-1" 1.0 0 -817084 true "" "plot count consumers with [ [ name ] of current-diet = \"flexitarian\" ] "
"pen-2" 1.0 0 -14070903 true "" "plot count consumers with [ [ name ] of current-diet = \"pescatarian\" ] "
"pen-3" 1.0 0 -15040220 true "" "plot count consumers with [ [ name ] of current-diet = \"vegetarian\" ] "
"pen-4" 1.0 0 -10141563 true "" "plot count consumers with [ [ name ] of current-diet = \"vegan\" ] "

SLIDER
205
440
430
473
mean-household-link-strength
mean-household-link-strength
0
1
0.925
0.001
1
NIL
HORIZONTAL

SLIDER
205
475
430
508
sd-household-link-strength
sd-household-link-strength
0
1
0.025
0.001
1
NIL
HORIZONTAL

SLIDER
435
440
635
473
mean-friends-link-strength
mean-friends-link-strength
0
1
0.825
0.001
1
NIL
HORIZONTAL

SLIDER
435
475
635
508
sd-friends-link-strength
sd-friends-link-strength
0
1
0.025
0.001
1
NIL
HORIZONTAL

SLIDER
205
510
430
543
mean-acquaintances-link-strength
mean-acquaintances-link-strength
0
1
0.35
0.001
1
NIL
HORIZONTAL

SLIDER
435
510
635
543
sd-acquaintances-link-strength
sd-acquaintances-link-strength
0
1
0.05
0.001
1
NIL
HORIZONTAL

SLIDER
655
480
875
513
mean-max-willingness-to-spend
mean-max-willingness-to-spend
0
1
0.16
0.01
1
NIL
HORIZONTAL

SLIDER
655
515
875
548
sd-max-willingness-to-spend
sd-max-willingness-to-spend
0
1
0.01
0.01
1
NIL
HORIZONTAL

SLIDER
20
250
195
283
mean-close-friends
mean-close-friends
0
initial-avg-degree
5.0
1
1
NIL
HORIZONTAL

SLIDER
20
465
195
498
taste-preference-change-error
taste-preference-change-error
0
1
0.001
0.001
1
NIL
HORIZONTAL

SLIDER
20
500
195
533
taste-preference-change-gradient
taste-preference-change-gradient
0
100
50.0
0.01
1
NIL
HORIZONTAL

SLIDER
20
320
195
353
link-strength-change-rate
link-strength-change-rate
0
1
0.001
0.001
1
NIL
HORIZONTAL

SLIDER
20
355
195
388
link-strength-change-gradient
link-strength-change-gradient
0
100
50.0
0.01
1
NIL
HORIZONTAL

SWITCH
1150
40
1330
73
social-interaction?
social-interaction?
0
1
-1000

SWITCH
1150
75
1330
108
network-structural-change?
network-structural-change?
0
1
-1000

TEXTBOX
1150
20
1320
38
For modular sensitivity analysis:
11
0.0
1

SWITCH
1150
110
1330
143
include-ethics-motivation?
include-ethics-motivation?
0
1
-1000

SWITCH
1150
145
1330
178
include-health-motivation?
include-health-motivation?
0
1
-1000

PLOT
885
305
1145
490
Flexitarian perceptions
NIL
NIL
-1.0
1.0
0.0
50.0
true
true
"" ""
PENS
"taste" 0.01 1 -2674135 true "histogram [ perceived-taste ] of consumer-diets with [ diet-name = \"flexitarian\" ]" "histogram [ perceived-taste ] of consumer-diets with [ diet-name = \"flexitarian\" ]"
"cost" 0.01 1 -7500403 true "histogram [ perceived-cost ] of consumer-diets with [ diet-name = \"flexitarian\" ]" "histogram [ perceived-cost ] of consumer-diets with [ diet-name = \"flexitarian\" ]"
"ethics" 0.01 1 -10899396 true "histogram [ perceived-ethics ] of consumer-diets with [ diet-name = \"flexitarian\" ]" "histogram [ perceived-ethics ] of consumer-diets with [ diet-name = \"flexitarian\" ]"
"health" 0.01 1 -13345367 true "histogram [ perceived-health ] of consumer-diets with [ diet-name = \"flexitarian\" ]" "histogram [ perceived-health ] of consumer-diets with [ diet-name = \"flexitarian\" ]"

SLIDER
655
405
875
438
mean-satisfaction-threshold
mean-satisfaction-threshold
0
1
0.5
0.001
1
NIL
HORIZONTAL

SLIDER
655
440
875
473
sd-satisfaction-threshold
sd-satisfaction-threshold
0
1
0.05
0.001
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Stability-test-new" repetitions="50" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;stability_test_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="no-health-ethics-motivation" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;baseline_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="no-ethics-motivation" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;baseline_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="diet-calibration" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="11"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;diet_calibration_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-mean-willingness-to-spend" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.08"/>
      <value value="0.098"/>
      <value value="0.116"/>
      <value value="0.133"/>
      <value value="0.151"/>
      <value value="0.169"/>
      <value value="0.187"/>
      <value value="0.204"/>
      <value value="0.222"/>
      <value value="0.24"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;baseline_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-willingness-to-spend" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.005"/>
      <value value="0.006"/>
      <value value="0.007"/>
      <value value="0.008"/>
      <value value="0.009"/>
      <value value="0.011"/>
      <value value="0.012"/>
      <value value="0.013"/>
      <value value="0.014"/>
      <value value="0.015"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;baseline_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-taste-pref-change-rate" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.005"/>
      <value value="0.006"/>
      <value value="0.007"/>
      <value value="0.008"/>
      <value value="0.009"/>
      <value value="0.011"/>
      <value value="0.012"/>
      <value value="0.013"/>
      <value value="0.014"/>
      <value value="0.015"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-taste-pref-change-error" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="5.0E-4"/>
      <value value="6.0E-4"/>
      <value value="7.0E-4"/>
      <value value="8.0E-4"/>
      <value value="9.0E-4"/>
      <value value="0.0011"/>
      <value value="0.0012"/>
      <value value="0.0013"/>
      <value value="0.0014"/>
      <value value="0.0015"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-n-consumers" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="125"/>
      <value value="153"/>
      <value value="181"/>
      <value value="208"/>
      <value value="236"/>
      <value value="264"/>
      <value value="292"/>
      <value value="319"/>
      <value value="347"/>
      <value value="375"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-mean-hh-link-strength" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.879"/>
      <value value="0.889"/>
      <value value="0.899"/>
      <value value="0.91"/>
      <value value="0.92"/>
      <value value="0.93"/>
      <value value="0.94"/>
      <value value="0.951"/>
      <value value="0.961"/>
      <value value="0.971"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-hh-link-strength" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.024"/>
      <value value="0.0242"/>
      <value value="0.0244"/>
      <value value="0.0247"/>
      <value value="0.0249"/>
      <value value="0.0251"/>
      <value value="0.0253"/>
      <value value="0.0256"/>
      <value value="0.0258"/>
      <value value="0.026"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-mean-friends-link-strength" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.784"/>
      <value value="0.793"/>
      <value value="0.802"/>
      <value value="0.811"/>
      <value value="0.82"/>
      <value value="0.83"/>
      <value value="0.839"/>
      <value value="0.848"/>
      <value value="0.857"/>
      <value value="0.866"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-friends-link-strength" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.024"/>
      <value value="0.0242"/>
      <value value="0.0244"/>
      <value value="0.0247"/>
      <value value="0.0249"/>
      <value value="0.0251"/>
      <value value="0.0253"/>
      <value value="0.0256"/>
      <value value="0.0258"/>
      <value value="0.026"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-mean-acquaint-link-strength" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.175"/>
      <value value="0.214"/>
      <value value="0.253"/>
      <value value="0.292"/>
      <value value="0.331"/>
      <value value="0.369"/>
      <value value="0.408"/>
      <value value="0.447"/>
      <value value="0.486"/>
      <value value="0.525"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-acquaint-link-strength" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.025"/>
      <value value="0.0306"/>
      <value value="0.0361"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-max-contacts" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
      <value value="11"/>
      <value value="12"/>
      <value value="13"/>
      <value value="14"/>
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-init-avg-degree" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="25"/>
      <value value="31"/>
      <value value="36"/>
      <value value="42"/>
      <value value="47"/>
      <value value="53"/>
      <value value="58"/>
      <value value="64"/>
      <value value="69"/>
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-mean-close-friends" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
      <value value="10"/>
      <value value="11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-link-strength-change-rate" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="5.0E-4"/>
      <value value="6.1E-4"/>
      <value value="7.2E-4"/>
      <value value="8.3E-4"/>
      <value value="9.4E-4"/>
      <value value="0.001056"/>
      <value value="0.001167"/>
      <value value="0.001278"/>
      <value value="0.001389"/>
      <value value="0.0015"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-rewiring-prob" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.25"/>
      <value value="0.306"/>
      <value value="0.361"/>
      <value value="0.417"/>
      <value value="0.472"/>
      <value value="0.528"/>
      <value value="0.583"/>
      <value value="0.639"/>
      <value value="0.694"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-soc-info-adherence-scaling" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.013"/>
      <value value="0.015"/>
      <value value="0.018"/>
      <value value="0.021"/>
      <value value="0.024"/>
      <value value="0.026"/>
      <value value="0.029"/>
      <value value="0.032"/>
      <value value="0.035"/>
      <value value="0.038"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-n-consumers-pt1" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="125"/>
      <value value="153"/>
      <value value="181"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-n-consumers-pt2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="208"/>
      <value value="236"/>
      <value value="264"/>
      <value value="292"/>
      <value value="319"/>
      <value value="347"/>
      <value value="375"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-acquaint-link-strength-pt2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.0417"/>
      <value value="0.0472"/>
      <value value="0.0528"/>
      <value value="0.0583"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-acquaint-link-strength-pt3" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.0639"/>
      <value value="0.0694"/>
      <value value="0.075"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="all-motivations" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;all_motivations_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="price-sensitivity-omnivore" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly_omnivore_10pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_omnivore_5pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_omnivore_5pct_increase.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_omnivore_10pct_increase.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;omnivore_price_sensitivity_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="price-sensitivity-flexitarian" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly_flexitarian_10pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_flexitarian_5pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_flexitarian_5pct_increase.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_flexitarian_10pct_increase.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;flexitarian_price_sensitivity_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="price-sensitivity-pescatarian" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly_pescatarian_10pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_pescatarian_5pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_pescatarian_5pct_increase.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_pescatarian_10pct_increase.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;pescatarian_price_sensitivity_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="price-sensitivity-vegetarian" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly_vegetarian_10pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_vegetarian_5pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_vegetarian_5pct_increase.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_vegetarian_10pct_increase.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;vegetarian_price_sensitivity_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="price-sensitivity-vegan" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly_vegan_10pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_vegan_5pct_decrease.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_vegan_5pct_increase.csv&quot;"/>
      <value value="&quot;input/final_prices_input_weekly_vegan_10pct_increase.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;vegan_price_sensitivity_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="baseline_params_no_struct_change" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;no_struct_change_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="baseline_params_no_interaction" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;no_interaction_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-mean-satisfaction-threshold" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.25"/>
      <value value="0.306"/>
      <value value="0.361"/>
      <value value="0.417"/>
      <value value="0.472"/>
      <value value="0.528"/>
      <value value="0.583"/>
      <value value="0.639"/>
      <value value="0.694"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-sd-satisfaction-threshold" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.025"/>
      <value value="0.031"/>
      <value value="0.036"/>
      <value value="0.042"/>
      <value value="0.047"/>
      <value value="0.053"/>
      <value value="0.058"/>
      <value value="0.064"/>
      <value value="0.069"/>
      <value value="0.075"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="OFAT-init-avg-degree-pt2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="58"/>
      <value value="64"/>
      <value value="69"/>
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;ofat_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="price-sensitivity-omnivore-reruns" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="157"/>
    <enumeratedValueSet variable="mean-satisfaction-threshold">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-satisfaction-threshold">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-contacts-per-timestep">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-error">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-max-willingness-to-spend">
      <value value="0.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-household-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-max-willingness-to-spend">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-avg-degree">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-close-friends">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-interaction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-friends-link-strength">
      <value value="0.825"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-cost-file">
      <value value="&quot;input/final_prices_input_weekly_omnivore_10pct_increase.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-param-output-file">
      <value value="&quot;omnivore_price_sensitivity_run_params.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-acquaintances-link-strength">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="taste-preference-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-gradient">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="soc-info-adherence-scaling-factor">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-consumers">
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="motivation-file">
      <value value="&quot;input/motivations_perceptions_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-ethics-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="household-file">
      <value value="&quot;input/household_input.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structural-change?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-acquaintances-link-strength">
      <value value="0.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="link-strength-change-rate">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="include-health-motivation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="my-seed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-information-adherence-file">
      <value value="&quot;input/norm_adherence_parameterization.csv&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-household-link-strength">
      <value value="0.925"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sd-friends-link-strength">
      <value value="0.025"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
