;;; creation of populations ;;;
breed [signals signal]                           ;; PQS (green) signalling molecules for iGEM setup
breed [signal2s signal2]                         ;; PQS (green) signalling molecules for WT setup
breed [PQSRs PQSR]                               ;; PQSR receptors bound to the inside membrane
breed [PQSAs PQSA]                               ;; PQSA promoter 
breed [GFPs GFP]                                 ;; GFP produced when the promoter is dimerised by interaction with BCAM0228P or PQS_2.PQSR

breed [BDSFs BDSF]                               ;; BDSF signalling molecule
breed [ATPs ATP]                                 ;; ATP inside cytoplasm
breed [ADPs ADP]                                 ;; ADPs inside cytoplasm that are formed when ATP loses a phosphate
breed [BCAM0228s BCAM0228]                       ;; BCAM0228 inside cytoplasm
breed [BCAM0228Ps BCAM0228P]                     ;; phosphorylated BCAM0228
breed [Pcblds Pcbld]                             ;; Pcbld promoter


;;; setup of all variables ;;;
globals [  
  i                                              ;; counts x coords when colouring cell membrane
  j                                              ;; counts y coords when colouring cell membrane
  x                                              ;; counts x coords when colouring cytoplasm
  y                                              ;; counts y coords when colouring cytoplasm
  
  difference                                     ;; count of the difference between ATP and ADP 

  a                                              ;; controls number of PQSRs/BCAM0227s for the top and bottom membrane
  b                                              ;; controls number of PQSRs/BCAM0227s for the side membranes
  c                                              ;; controls number of PQSRs for the corners of the membrane
  sep                                            ;; controls distance between PQSRs/BCAM0227s on the top and bottom membrane
  sep2                                           ;; controls distance between PQSRs/BCAM0227s for the side membranes
  ja                                             ;; counts x coords for PQSR/BCAM0227 setup
  jap                                            ;; counts y coords for PQSR/BCAM0227 setup
  
  degraded-signal                                ;; count signals degraded
  degraded-GFP                                   ;; count GFPs degraded
  todie                                          ;; count signals to die each tick
]

;;; attributes associated with all the turtles ;;;
turtles-own [
  speed mass energy                              ;; controls the speed, mass and energy of the turtles for kinetics
  last-collision                                 ;; keeps note of when the previous collision has occured
  sticker                                        ;; boolean variable keeping track of whether a PQS is bound to PQSRPqsR  
  dimerised                                      ;; keeps note of whether PQSR has been dimerised by signalling molecules
  ready                                          ;; boolean variable keeping track of whether the promoter is interacting with the activated receptor
  bound-signal                                   ;; a counter for the number of signalling molecules bound to PQSR
  BDSF-bound                                     ;; boolean variable for whether BDSF is bound to BCAM0227
  transporter                                    ;; boolean variable for transporting the newly synthesized PQS out of the cell
  ATP-bound                                      ;; boolean variable for whether ATP is bound to BCAM0227 
  BCAM-bound                                     ;; boolean vairable for whether BCAM0228 is bound to BCAM027P 
  ]

to make-movie-WT-PQS
  user-message "First, save your new movie file (choose a name ending with .mov)"                     ;; prompt user for movie location
  let path user-new-file 
  if not is-string? path [ stop ]  ;; stop if user canceled
  setup-WT-PQS
  movie-start path
  movie-grab-view
  while [count signal2s with [color = sky] < 25 ]
    [ go
      movie-grab-view ]
  movie-close                                                                                          ;; export the movie
  user-message (word "Exported movie to " path)
end

to make-movie-iGEM-PQS
  user-message "First, save your new movie file (choose a name ending with .mov)"                     ;; prompt user for movie location
  let path user-new-file 
  if not is-string? path [ stop ]  ;; stop if user canceled
  setup-iGEM-PQS
  movie-start path
  movie-grab-view
  while [ ticks <= 1000]
    [ go
      movie-grab-view ]
  movie-close                                                                                          ;; export the movie
  user-message (word "Exported movie to " path)
end

to make-movie-iGEM-BDSF
  user-message "First, save your new movie file (choose a name ending with .mov)"                     ;; prompt user for movie location
  let path user-new-file 
  if not is-string? path [ stop ]  ;; stop if user canceled
  setup-iGEM-BDSF
  movie-start path
  movie-grab-view
  while [count GFPs >= 0 and count GFPs < 10 ]
    [ go
      movie-grab-view ]
  movie-close                                                                                          ;; export the movie
  user-message (word "Exported movie to " path)
end


;;; creates a class called particles for calling-in GFPs, ADPs, BCAM0228s and BCAM0228Ps ;;;
to-report particles
  report (turtle-set GFPs ADPs BCAM0228s BCAM0228Ps)
end

;;; setup the characteristics of PQS for the iGEM setup ;;;
to setup-signal
   set speed 10
   set mass 25000                                    
   set energy (0.5 * mass * (speed ^ 2))
   set last-collision nobody
   set size 4
   set shape "dot"
   set sticker false
   set bound-signal 0
end

;;; setup the characteristics of PQS molecules for the WT setup ;;;
to setup-signal2
   set speed 10
   set mass 25000                                    
   set energy (0.5 * mass * (speed ^ 2))
   set last-collision nobody
   set size 4
   set shape "dot"
   set sticker false
   set bound-signal 0
   set transporter false
end

;;; setup the characteristics for PQSA promoters ;;;
to setup-PQSA
   set speed 10
   set mass 25000                                    
   set energy (0.5 * mass * (speed ^ 2))
   set last-collision nobody
   set size 4
   set shape "circle 2"
   set color 78
   set ready false  
end

;;; setup the characteristics for PQSR receptors ;;;
to setup-PQSR
  set color 103 
  set shape "square"
  set size 4
end

;;; setup the characteristics for the GFP molecules ;;;
to setup-GFP
  set speed 5
  set mass 1000
  set energy (0.5 * mass * (speed ^ 2))
  set last-collision nobody
  set color red
  set size 5
  set shape "star"
end

;;; setup the characteristics for the of BDSF molecules ;;;
to setup-BDSF
  set speed 5
  set mass 1500
  set energy (0.5 * mass * (speed ^ 2))
  set last-collision nobody
  set color gray
  set size 3
  set shape "dot"
  set BDSF-bound false
end

;;; setup the characteristics for the of ATP molecules ;;;
to setup-ATP
   set speed 5
   set mass 100
   set energy (0.5 * mass * (speed ^ 2))
   set last-collision nobody
   set color 26
   set size 2
   set shape "dot"
   set ATP-bound false
end

;;; setup the characteristics for the of ADP molecules ;;;
to setup-ADP
   set speed 5
   set mass 100
   set energy (0.5 * mass * (speed ^ 2))
   set last-collision nobody
   set color 22
   set size 2
   set shape "dot"
end

;;; setup the characteristics for the of BCAM0228 molecules ;;;
to setup-BCAM0228
  set speed 5
  set mass 500
  set energy (0.5 * mass * (speed ^ 2))
  set last-collision nobody
  set color 106
  set size 3
  set shape "square"
  set BCAM-bound false
end

;;; setup the characteristics for the of BCAM0228P molecules ;;;
to setup-BCAM0228P
  set speed 5
  set mass 500
  set energy (0.5 * mass * (speed ^ 2))
  set last-collision nobody
  set color red
  set size 3
  set shape "square"
end

;;; setup the characteristics for Pcbld promoters ;;;
to setup-Pcbld
  set speed 5
  set mass 800
  set energy (0.5 * mass * (speed ^ 2))
  set last-collision nobody
  set color 78
  set size 3
  set shape "circle 2"
end

;;;;; setup for iGEM PQS system ;;;;;
to setup-iGEM-PQS
  clear-all
  
;;; setup the initial values for the loops ;;;  
  set i 30
  set j 29
  set x 31
  set y 25
  set ja (-30) 
  set jap (-10)
  
;;; calls for the creation of the environment ;;;  
  background
  
;;; creation of the cell membrane ;;;
  top-right-corner
  set i -30
  set j 29
  top-left-corner
  set i -30
  set j -29
  bottom-left-corner
  set i 30
  set j -29
  bottom-right-corner
  
;;; colouring in the cytoplasm ;;;
  cytoplasm-top
  set x 31
  set y -25
  cytoplasm-bottom  
  

  
;;; creation of the other molecules, the number of each is controlled by a slider ;;;
;;; with random positions either outside the cell (randomize-extracellular) or within the cell (randomize-intracellular) ;;;
  create-signals initial-PQS
  [ setup-signal
    set color green
    randomize-extracellular]
  
  create-PQSAs initial-PQSA
  [ setup-PQSA
    randomize-intracellular]
  
  ;;; creation the PQSR receptors ;;;
  receptors
  
  reset-ticks
end

;;;;; setup for WT PQS system ;;;;;
to setup-WT-PQS
  clear-all
  
;;; setup the initial values for the loops ;;;  
  set i 30
  set j 29
  set x 31
  set y 25
  set ja (-30) 
  set jap (-10)
  
;;; calls for the creation of the environment ;;;  
  background
  
;;; creation of the cell membrane ;;;
  top-right-corner
  set i -30
  set j 29
  top-left-corner
  set i -30
  set j -29
  bottom-left-corner
  set i 30
  set j -29
  bottom-right-corner
  
;;; colouring in the cytoplasm ;;;
  cytoplasm-top
  set x 31
  set y -25
  cytoplasm-bottom  
  
;;; creation the PQSR receptors ;;;
  receptors
  
;;; creation of the other molecules, the number of each is controlled by a slider ;;;
;;; with random positions either outside the cell (randomize-extracellular) or within the cell (randomize-intracellular) ;;;
  create-signal2s initial-PQS
  [ setup-signal
    set color green
    randomize-extracellular]
  
  create-PQSAs initial-PQSA
  [ setup-PQSA
    randomize-intracellular]
  
  reset-ticks
end

;;;;; setup for iGEM BDSF system ;;;;;
to setup-iGEM-BDSF
  clear-all
  
;;; setup the initial values for the loops ;;;    
  set i 30
  set j 29
  set x 31
  set y 25

set ja -31
set jap -10
  
;;; calls for the creation of the environment ;;;   
  background
  
;;; creation of the cell membrane ;;;
  top-right-corner
  set i -30
  set j 29
  top-left-corner
  set i -30
  set j -29
  bottom-left-corner
  set i 30
  set j -29
  bottom-right-corner
  
;;; colouring in the cytoplasm ;;;
  cytoplasm-top
  set x 31
  set y -25
  cytoplasm-bottom  
  
BCAM0227
  
;;; creation of the other molecules, the number of each is controlled by a slider ;;;
;;; with random positions either outside the cell (randomize-extracellular) or within the cell (randomize-intracellular) ;;;  
  create-BDSFs initial-BDSF
  [ setup-BDSF
    randomize-extracellular]
  
  create-BCAM0228s initial-BCAM0228
  [ setup-BCAM0228
    randomize-intracellular]
  
  create-Pcblds initial-Pcbld
  [ setup-Pcbld
    randomize-intracellular]
  
  create-ATPs initial-ATP
  [ setup-ATP
    randomize-intracellular]
  
  reset-ticks
end

;;; position randomizing procedure for extracellular particles (PQS,BDSF) ;;;
to randomize-extracellular
 setxy random-xcor random-ycor
  if (pcolor != 8) 
  [randomize-extracellular] 
end

;;; position randomzing procedure for intracellular particles ;;;
to randomize-intracellular
 setxy random-xcor random-ycor
  if (pcolor != 78) 
  [randomize-intracellular] 
end

;;;;; this procedure is what is run through (looping) during the operation of the simulation ;;;;;
to go
  if ticks > 1200 [stop]
  
  ;;; for all turtles except PQSRs check for collision and apply random motion ;;;
  ask turtles [
    if breed != PQSRs [
      check-for-collision
      rt random-float 360]]
  
  ;;; signalling molecules must only continue moving if they are not bound to PQSR ;;;
  ask signals [
    if sticker = false [
      stick
      bounce
      fd 1]] 
  
  ;;; signalling molecules must only continue moving if they are not bound to PQSR ;;;
  ask signal2s [
    if sticker = false [
      stick2
      bounce
      fd 1] 
    if transporter = true [
      transport]] 
  
  ;;; the promoters can flock towards to the receptors and bounce off the membrane ;;;
  ask PQSAs [
    flock
    bounce
    fd 1]
  
  ;;; BDSFs can only continue moving if they are not bound to BCAM0227 ;;;
  ask BDSFs [
    bounce
    if BDSF-bound = false [
      fd 1 ]]
  
  ;;; ATPs can only continue moving if they are not bound to BCAM0227 ;;;
  ask ATPs [
    bounce
    if ATP-bound = false [
      fd 1 ]]
  
  ;;; these promoters can bounce off the membrane or bind to BCAM0228Ps ;;;
  ask Pcblds [
    bounce
    bind
    fd 1]
  
  ;;; particles can bounce off the membrane ;;;
  ask particles [   
    bounce 
    fd 1]
  
  ;;; decay procedure
 degrade
  
  ;;; in order to keep the number of ATPs high, create more ;;;
  if count ADPs > (initial-ATP / 5) [                                ;; if the count of ADPs is greater than a fifth of the initial-ATP count
    set difference (count ADPs - (initial-ATP / 5))                  ;; set the global variable "difference" to the difference between the two values
    create-ATPs difference                                           ;; create "difference" number of ATPs
    [ setup-ATP                                                  
      randomize-intracellular]                                
    ask n-of difference ADPs [die]                                   ;; ask "difference" number of ADPs to die
    set difference 0]                                                ;; reset the global variable "difference" to 0
 
  ;;; the simulation continues forward one tick ;;; 
  tick
end

;;; procedure to create the background of the simulation ;;;
to background
  ask patches with [pxcor >= -60 and pxcor <= 60][set pcolor 8]                                              ;; colour the extracellular region light grey 
  ask patches with [pycor > 25 and pycor < 30 and pxcor > -30 and pxcor < 30][set pcolor 74]                 ;; colour the top part of the membrane dark turquoise
  ask patches with [pycor > -30 and pycor < -25 and pxcor > -30 and pxcor < 30][set pcolor 74]               ;; colour the bottom part of the membrane dark turquoise
  ask patches with [pycor > -11 and pycor < 11 and pxcor > 47 and pxcor < 52][set pcolor 74]                 ;; colour the rhs part of the membrane dark turquoise
  ask patches with [pycor > -11 and pycor < 11 and pxcor > -52 and pxcor < -47][set pcolor 74]               ;; colour the lhs part of the membrane dark turquoise
end

;;; procedures to create the curved corners of the cell membrane  

to top-right-corner                                                                                          ;; for the top right corner of the cell membrane
  ask patches with [pxcor = i and pycor = j][                                                                ;; set a global variable "i" for the xcoord and a "j" for the ycoor                                                                                                                                
    if i < 40 [                                                                                              ;; the curve has to be created in stages 
      ask patches with [pxcor >= i and pxcor < (i + 2) and pycor <= j and pycor > (j - 4)][set pcolor 74]    ;; for the first section we want the curve to be gentle with depth 4 patches
      set i (i + 2)                                                                                          ;; the general outline for this section is move along right 2 
      set j (j - 1)                                                                                          ;; and down 1
      top-right-corner]                                                                                      ;; repeat until section is complete
    if i >= 40 and i < 48 [                                                                                  ;; the next section is slightly steeper 
      ask patches with [pxcor = i and pycor <= j and pycor > (j - 5)][set pcolor 74]                         ;; this section has depth 5 to compensate for the inside edge being smaller than the other
      set i (i + 1)                                                                                          ;; general outline here being move along right 1
      set j (j - 1)                                                                                          ;; and down 1
      top-right-corner]                                                                                      ;; repeat until section is complete
    if i = 48 [                                                                                              ;; the final section is a little different and so is written in 3 parts 
      ask patches with [pxcor = i and pycor <= j and pycor > (j - 6)][set pcolor 74]                         ;; each part has to stop inline with the right side membrane
      set i (i + 1)                                                                                          ;; so the 3 parts follow the outline move along 1 right
      set j (j - 2)                                                                                          ;; and move down 2
      top-right-corner]                                                                                      ;; but each part has a depth 2 shorter than the previous 
    if i = 49 [                                                                                              ;; so at i = 48, the depth is 6 so membrane is betweeen j and (j - 6)
      ask patches with [pxcor = i and pycor <= j and pycor > (j - 4)][set pcolor 74]                         ;; at i = 49, the depth is 4 so membrane is between j and (j - 4) 
      set i (i + 1)                                                                                          ;; and finally at i = 50, the depth is 2 so membrane is between j and (j - 2)
      set j (j - 2)                          
      top-right-corner]
    if i = 50 [
      ask patches with [pxcor = i and pycor <= j  and pycor > (j - 2)][set pcolor 74]]
    ask patches with [pxcor = 47 and (pycor = 11 or pycor = 12)][set pcolor 74]                              ;; these last 3 patch commands are to make the curve look smoother
    ask patches with [pxcor = 46 and pycor = 13][set pcolor 74]
    ask patches with [pxcor = 39 and pycor = 21][set pcolor 74]]
end
   
to top-left-corner                                                                                           ;; for the top left corner of the cell membrane
  ask patches with [pxcor = i and pycor = j][                                                                ;; procedure follows the same protocol as top-right-corner                                                                        
  if i > -40 [                                                                                               ;; just with different coordinates
    ask patches with [pxcor <= i and pxcor > (i - 2) and pycor <= j and pycor > (j - 4)][set pcolor 74]
    set i (i - 2)
    set j (j - 1)
    top-left-corner]
  if i <= -40 and i > -48 [
    ask patches with [pxcor = i and pycor <= j and pycor > (j - 5)][set pcolor 74]
    set i (i - 1)
    set j (j - 1)
    top-left-corner]
  if i = -48 [
    ask patches with [pxcor = i and pycor <= j and pycor > (j - 6)][set pcolor 74]
    set i (i - 1)
    set j (j - 2)
    top-left-corner]
  if i = -49 [
    ask patches with [pxcor = i and pycor <= j and pycor > (j - 4)][set pcolor 74]
    set i (i - 1)
    set j (j - 2)
    top-left-corner]
  if i = -50 [
    ask patches with [pxcor = i  and pycor <= j and pycor > (j - 2)][set pcolor 74]
    set i (i - 1)
    set j (j - 2)
    top-left-corner]
  ask patches with [pxcor = -47 and (pycor = 11 or pycor = 12)][set pcolor 74]
  ask patches with [pxcor = -46 and pycor = 13][set pcolor 74]
  ask patches with [pxcor = -39 and pycor = 21][set pcolor 74]]
end
 
to bottom-left-corner                                                                                        ;; for the bottom left corner of the cell membrane
  ask patches with [pxcor = i and pycor = j][                                                                ;; procedure follows the same protocol as top-right-corner  
  if i > -40 [                                                                                               ;; just with different coordinates
    ask patches with [pxcor <= i and pxcor > (i - 2) and pycor >= j and pycor < (j + 4)][set pcolor 74]
    set i (i - 2)
    set j (j + 1)
    bottom-left-corner]
  if i <= -40 and i > -48 [
    ask patches with [pxcor = i and pycor >= j and pycor < (j + 5)][set pcolor 74]
    set i (i - 1)
    set j (j + 1)
    bottom-left-corner]
  if i = -48 [
    ask patches with [pxcor = i and pycor >= j and pycor < (j + 6)][set pcolor 74]
    set i (i - 1)
    set j (j + 2)
    bottom-left-corner]
  if i = -49 [
    ask patches with [pxcor = i and pycor >= j and pycor < (j + 4)][set pcolor 74]
    set i (i - 1)
    set j (j + 2)
    bottom-left-corner]
  if i = -50 [
    ask patches with [pxcor = i and pycor >= j and pycor < (j + 2)][set pcolor 74]
    set i (i - 1)
    set j (j + 2)
    bottom-left-corner]
  ask patches with [pxcor = -47 and (pycor = -11 or pycor = -12)][set pcolor 74]
  ask patches with [pxcor = -46 and pycor = -13][set pcolor 74]
  ask patches with [pxcor = -39 and pycor = -21][set pcolor 74]]
end
   
to bottom-right-corner                                                                                       ;; for the bottom left corner of the cell membrane
  ask patches with [pxcor = i and pycor = j][                                                                ;; procedure follows the same protocol as top-right-corner
    if i < 40 [                                                                                              ;; just with different coordinates
      ask patches with [pxcor >= i and pxcor < (i + 2) and pycor >= j and pycor < (j + 4)][set pcolor 74]
      set i (i + 2)
      set j (j + 1)
      bottom-right-corner]
    if i >= 40 and i < 48 [
      ask patches with [pxcor = i and pycor >= j and pycor < (j + 5)][set pcolor 74]
      set i (i + 1)
      set j (j + 1)
      bottom-right-corner]
    if i = 48 [
      ask patches with [pxcor = i and pycor >= j and pycor < (j + 6)][set pcolor 74]
      set i (i + 1)
      set j (j + 2)
      bottom-right-corner]
    if i = 49 [
      ask patches with [pxcor = i and pycor >= j and pycor < (j + 4)][set pcolor 74]
      set i (i + 1)
      set j (j + 2)
      bottom-right-corner]
    if i = 50 [
      ask patches with [pxcor = i and pycor >= j and pycor < (j + 2)][set pcolor 74]
      set i (i + 1)
      set j (j + 2)
      bottom-right-corner]
    ask patches with [pxcor = 47 and (pycor = -11 or pycor = -12)][set pcolor 74]
    ask patches with [pxcor = 46 and pycor = -13][set pcolor 74]
    ask patches with [pxcor = 39 and pycor = -21][set pcolor 74]] 
end

;; commands to color the cytoplasm
;; due to the curved corners of the cell membrane the cytoplasm has to be "colored" pretty much line by line based on coordinates 
 
to cytoplasm-top                                                                                           ;; for the top half of the cytoplasm
if y > 22 and y <= 25 [                                                                                    ;; the global variables are "x" for the xcoord and "y" for the ycoord
 ask patches with [pxcor >= (- x) and pxcor <= x and pycor = y][set pcolor 78]                             ;; since the cell has vertical symmetry the area to be coloured is between x and -x 
   set x (x + 2)                                                                                           ;; the first part follows the gentle curve of the cell membrane and so the general outline for this section
   set y (y - 1)                                                                                           ;; move down 1 ycoord and add 2 xcoord
   cytoplasm-top]                                                                                          ;; repeat until section is complete
if y > 14 and y <= 22 [                                                                                    ;; the next section has 2 colours
  ask patches with [pxcor >= (- x) and pxcor <= (x) and pycor = y][set pcolor 78]                          ;; 77.9 is the region closest to the membrane of width 5 patches and 78 (marginally lighter) is the centre cytoplasm region                  ;; this colour difference is to contain the signalling molecules in the region closest to the membrane ( 77.9 colour)
   set x (x + 1)                                                                                           ;; the general outline for this section is down 1 and add 1 x since th curve is steeper
   set y (y - 1)
   cytoplasm-top]                                                                                          ;; repeat until section is complete 
if y > 10 and y <= 14 [                                                                                    ;; the curve of the next section is steeper still and so this section follows the outline down 2 and add 1 x
  ask patches with [pxcor >= (- x) and pxcor <= (x) and (pycor = y or pycor = (y - 1))][set pcolor 78]
   set x (x + 1)
   set y (y - 2)
   cytoplasm-top]    
if y >= 0 and y <= 10 [                                                                                    ;; in the final section the membrane is no longer curved
  ask patches with [                                                                                       ;; and so the general outline is just down 1 ycoord
    pxcor >= (- x) and pxcor <= x and pycor = y][set pcolor 78]
  set y ( y - 1)
  cytoplasm-top]                                                                                          
ask patches with [pycor > -11 and pycor < 11 and pxcor > 44 and pxcor <= 47][set pcolor 78]              ;; as before the region closest to the membrane as to be slightly darker than the main cytomplasmic region
ask patches with [pycor > -11 and pycor < 11 and pxcor >= -47 and pxcor < -44][set pcolor 78]
end  

to cytoplasm-bottom                                                                                        ;; the bottom half of the cytoplasm is coloured in the same way as the top half
if y < -22 and y >= -25 [                                                                                  ;; but starts at the very bottom of the cell
  ask patches with [                                                                                       ;; and so the procedure runs through increading ycoords instead of decreasing ycoords!
 pxcor >= (- x) and pxcor <= x and pycor = y][set pcolor 78]
   set x (x + 2)
   set y (y + 1)
   cytoplasm-bottom] 
if y < -14 and y >= -22 [
  ask patches with [pxcor >= (- x) and pxcor <= (x) and pycor = y][set pcolor 78]
   set x (x + 1)
   set y (y + 1)
   cytoplasm-bottom] 
if y < -10 and y >= -14 [
  ask patches with [pxcor >= (- x) and pxcor <= (x) and (pycor = y or pycor = (y + 1))][set pcolor 78]
   set x (x + 1)
   set y (y + 2)
   cytoplasm-bottom] 
if y < 0 and y >= -10 [ 
  ask patches with [
    pxcor >= (- x) and pxcor <= x and pycor = y][set pcolor 78]
  set y ( y + 1)
  cytoplasm-bottom]
ask patches with [pycor > -11 and pycor < 11 and pxcor > 44 and pxcor <= 47][set pcolor 78]
ask patches with [pycor > -11 and pycor < 11 and pxcor >= -47 and pxcor < -44][set pcolor  78]
end  

;;; procedure for the PQSR receptors in the PQS system ;;; 
;;; the receptors sit on the inside part of the membrane ;;;
;;; we need a procedure which spreads the receptors equally around the cell ;;;
;;; in an actual cell the receptors can move but for the purpose of this animation they have a fixed location ;;;

to receptors                  ;; a slider specifies the initial number of PQSR receptors.  This number is a multiple of 4 between 4 and 52
  set a ((initial-PQSR / 4))  ;; The initial number of PQSR has to split between the top/bottom, sides and corners of the membrane
  set b (initial-PQSR / 8)    ;; in general the top/bottom get 1/4 of initial-PQSR each; the sides get 1/8th each; and corners 1/16th each
                
  if initial-PQSR != 4 and initial-PQSR != 8 [    ;; if the initial-PQSR is 4 or 8 the positioning is slightly different                       
    
    ;;;  for the top/bottom of the membrane ;;; 
    set sep (floor(60 /( a * 2)))               
    if ja <= (30 - sep) [                              
      ask patches with [(pycor = 25  and pxcor = (ja + sep)) or (pycor = -25  and pxcor = (ja + sep)) ]                       
        [ sprout-PQSRs 1 [setup-PQSR]]
      set ja (ja + (2 * sep) )
      receptors]
   
   ;;; for the sides of the membrane ;;; 
    set sep2 (ceiling (20 /( b * 2)))
    if jap <= (10 - sep2) [                              
      ask patches with [(pycor = (jap + sep2)  and pxcor = 47) or (pycor = (jap + sep2)  and pxcor = -47) ]    
        [ sprout-PQSRs 1 [setup-PQSR]]
      set jap (jap + (2 * sep2))
      receptors]]
  
  ;;; if the number of PQSRs on the top/bottom and the sides is less than initial-PQSRs we make up the numbers on the corners 
  if count PQSRs < initial-PQSR [
    set c (initial-PQSR - (count PQSRs))
    if  c != 8 [
      ask patches with [pxcor = 40 and pycor = 19][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = -40 and pycor = -19][ sprout-PQSRs 1 [setup-PQSR]]]
    if c = 4 or c = 12[   
      ask patches with [pxcor = -40 and pycor = 19][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = 40 and pycor = -19][ sprout-PQSRs 1 [setup-PQSR]]]
    
    if c >= 6[
      ask patches with [pxcor = -36 and pycor = 23][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = -44 and pycor = 15][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = 36 and pycor = -23][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = 44 and pycor = -15][ sprout-PQSRs 1 [setup-PQSR]]]
    
    if c >= 8[
      ask patches with [pxcor = -36 and pycor = -23][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = -44 and pycor = -15][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = 36 and pycor = 23][ sprout-PQSRs 1 [setup-PQSR]]
      ask patches with [pxcor = 44 and pycor = 15][ sprout-PQSRs 1 [setup-PQSR]]]
  ]
end
  
to BCAM0227                       ;; a slider specifies the initial number of BCAM0227.  This number is a multiple of 6 between 6 and 30
  set a ((initial-BCAM0227 / 3))  ;; The initial number of BCAM0227 has to split between the top/bottom and sides of the membrane
  set b (initial-BCAM0227 / 6)    ;; in general the top/bottom get 1/3rd of initial-BCAM0227 each and the sides get 1/6th each
  
  ;;;  for the top/bottom of the membrane ;;; 
  set sep (ceiling(60 / (a * 2)))             
  if ja <= (31 - sep) [                              
    ask patches with [(pxcor = (ja + sep) and (pycor > 23 and pycor < 32)) or (pxcor = (ja + sep) and (pycor < -23 and pycor > -32)) ]                       
      [ set pcolor 125]
    set ja (ja + (2 * sep))
    BCAM0227]
  
  ;;; for the sides of the membrane ;;; 
  set sep2 (round (20 /( b * 2 )))
  ifelse initial-BCAM0227 != 18 [
    if jap <= (11 - sep2) [                              
      ask patches with [(pycor = (jap + sep2) and (pxcor > 45 and pxcor < 54)) or (pycor = (jap + sep2)  and (pxcor < -45 and pxcor > -54)) ]    
        [ set pcolor 125]
      set jap (jap + (2 * sep2))
      BCAM0227]]
  [ if jap <= (10 - sep2) [                              
    ask patches with [(pycor = (jap + sep2) and (pxcor > 45 and pxcor < 54)) or (pycor = (jap + sep2)  and (pxcor < -45 and pxcor > -54)) ]    
      [ set pcolor 125]
    set jap (jap + (2 * sep2))
    BCAM0227]]
end
    
;;; procedure for interaction of particles and cell structures ;;;
to bounce 
  ;; signalling molecules can diffuse through the membrane or bounce off it ;;
  if breed = signals or breed = signal2s[
    if random 100 >= diffuseprob [  
      if [pcolor] of patch-at dx 0 = 74 [set heading (- heading)]
      if [pcolor] of patch-at 0 dy = 74 [set heading (180 - heading)]]]
  
  ;; when a BDSF molecule interacts with a BCAM0227 it stays bound. The BCAM0227 will change colour from 125 (magenta) to 82 (dark cyan) ;;
  ;; once the BCAM0227 has transferred its phosphate to BCAM0228 the BDSF dissociates ;; 
  if breed = BDSFs [
    if [pcolor] of patch-here = 125 [set BDSF-bound true activation]
    if [pcolor] of patch-here = 124 [set BDSF-bound false deactivate]]
  
  ;; when an ATP molecule interacts with a BDSF.BCAM0227 it phosphorylates the BCAM0277. The BCAM0227 will change colour from 82 (dark cyan) to red ;;
  if breed = ATPs [
    if [pcolor] of patch-here = 82 [set ATP-bound true autophosphorylation]]
  
  ;; when BCAM0228 interacts with a phosphorylated BCAM0227 the phosphate is transferred from BCAM0227 to BCAM0228. BCAM0228 changes from blue to red and BCAM0227 from red to 124 (dark magenta)
  if breed = BCAM0228s [
    if [pcolor] of patch-here = red [set BCAM-bound true steal]]
 
  ;; all turtles except ATP can bounce off BDSF.BCAM0227 complex (82 - dark cyan) ;; 
  if breed != ATPs [
    if [pcolor] of patch-at dx 0 = 82 [set heading (- heading)]
    if [pcolor] of patch-at 0 dy = 82 [set heading (180 - heading)]]
  
  ;; all turtles except BCAM0228 can bounce off a phosphorylated BCAM0227 (red) ;;
  if breed != BCAM0228s [
    if [pcolor] of patch-at dx 0 = red [set heading (- heading)]
    if [pcolor] of patch-at 0 dy = red [set heading (180 - heading)]]
  
  ;; all turtles except BDSF bounce off BCAM0227 (pcolor 125) and unphosphorylate BCAM0227 (pcolor 124) ;;
  if breed != BDSFs [
    if [pcolor] of patch-at dx 0 = 124 [set heading (- heading)]
    if [pcolor] of patch-at 0 dy = 124 [set heading (180 - heading)]
    
    if [pcolor] of patch-at dx 0 = 125 [set heading (- heading)]
    if [pcolor] of patch-at 0 dy = 125 [set heading (180 - heading)]]
  
  ;; all turtles except PQS (since they have special instructions) bounce off the membrane ;;
  if breed != signals and breed != signal2s [
    if [pcolor] of patch-at dx 0 = 74 [set heading (- heading)]
    if [pcolor] of patch-at 0 dy = 74 [set heading (180 - heading)]]
end

;;; procedure for binding 2 signalling molecules to PQSR in iGem setup ;;;
to stick
  let receptor one-of other PQSRs-here
  if receptor != nobody [
    ask receptor [
      if bound-signal < signal-required [
        if random 100 <= (bindprob) [                  
          ask signals-here [
            set sticker true ]
          set bound-signal (bound-signal + 1 )]
      ]]]
end

;;; procedure for binding 2 signalling molecules to PQSR in wt setup ;;;
to stick2 
  let receptor one-of other PQSRs-here
  if receptor != nobody [
    ask receptor [
      if bound-signal < signal-required [
        if random 100 <= (bindprob) [
          ask signal2s-here [
            set sticker true ]
          set bound-signal (bound-signal + 1 )] 
      ]]]
end

;;; procedure for the promoter to be attracted to the PQSR which has been dimerisized by PQS ;;;
to flock
  set dimerised one-of PQSRs with [bound-signal = 2] 
  if dimerised != nobody [
    ask dimerised [
      let candidate min-one-of PQSAs [distance myself]
      ask candidate [ 
        face myself  
        set ready true
        transcribe
      ]]]
end

;;; procedure from promoter and activated PQSR interaction ;;;
to transcribe 
  ;; in iGem setup GFP is produced ;;
  if count PQSAs-here = 1 and ready = true and count PQSRs-here = 1 and count signals in-radius 2 = 2 [
    ask signals in-radius 2 [
      move-to one-of patches with[ pcolor = 8]
      set sticker false
      set bound-signal 0 ]
    hatch-GFPs 1 [                           
      setup-GFP]
    ask PQSRs-here [
      set bound-signal 0] 
    ask PQSAs-here [
      set ready false]] 
  
  ;; in wt setup PQS is produced. However it is coloured sky blue to differentiate between initial PQS and new PQS ;;
  if count PQSAs-here = 1 and ready = true and count PQSRs-here = 1 and count signal2s in-radius 2 = 2 [
    ask signal2s in-radius 2 [
      move-to one-of patches with[ pcolor = 8]
      set sticker false
      set bound-signal 0 ] 
    hatch-signal2s 1 [
      setup-signal2
      set color sky
      set transporter true]
    ask PQSRs-here [
      set bound-signal 0]
    ask PQSAs-here [
      set ready false]] 
end

;;; procedure for movement of new PQS out of the cell ;;;
;;; the new PQS can't bind to PQSR until it has moved out of the cell and back in again ;;;
to transport
  ifelse random 100 <= diffuseprob [  
    face one-of patches with [pcolor = 8]
    if [pcolor] of patch-here = 8 [ set transporter false
      set sticker false
      set bound-signal 0]]
  [if [pcolor] of patch-at dx 0 = 74 [set heading (- heading)]
    if [pcolor] of patch-at 0 dy = 74 [set heading (180 - heading)]]
  fd 1
end

;;; procedure to allow BDSF to attach to BCAM0227 and change the colour of BCAM0227 to dark grey ;;;
to activation
  ask patches in-radius 1 with [pcolor = 125] [
    set pcolor 82]
  repeat (1.5 * 12) [
    ask patches with [pcolor = 82] [
      ask patches in-radius 1 with [pcolor = 125] [
        set pcolor 82]]]
end

;;; procedure for ATP to transfer a phosphate to the complex BDSF.BCAM0227 and change the breed to ADP and change the colour of BDSF.BCAM0227 to red ;;;
to autophosphorylation
  if ATP-bound = true [
    set breed ADPs
    setup-ADP 
    
    ask patches in-radius 1 with [pcolor = 82] [
      set pcolor red]
    repeat (1.5 * 12) [
      ask patches with [pcolor = red] [
        ask patches in-radius 1 with [pcolor = 82] [
          set pcolor red]]]
    set ATP-bound false]
end


;;; procedure for BCAM0228 to "steal" phosphate from the complex BDSF.BCAM0227 and change breed to BCAM0228P and BDSF.BCAM0227 color to dark magenta (124) ;;;
to steal
  if BCAM-bound = true [
    set breed BCAM0228Ps
    face one-of patches with [pcolor = 78]
    setup-BCAM0228P
    
    ask patches in-radius 1 with [pcolor = red] [
      set pcolor 124]
    repeat (1.5 * 12) [
      ask patches with [pcolor = 124] [
        ask patches in-radius 1 with [pcolor = red] [
          set pcolor 124]]]]
end

;;; procedure to change BCAM0227 back into a state where BDSF can join to it - BDSF detaches from BCAM0227 changing BCAM0227 colour to magenta (125) ;;;
to deactivate
  ask patches in-radius 1 with [pcolor = 124] [
    set pcolor 125]
  repeat (1.5 * 12) [
    ask patches with [pcolor = 125] [
      ask patches in-radius 1 with [pcolor = 124] [
        set pcolor 125]]]
  set BDSF-bound false
  move-to one-of patches with[ pcolor = 8]
end

;;; procedure to synthesize GFP when BCAM0228P interacts with Pcbld ;;;
to bind                                                                    
  ask BCAM0228Ps-here [
    hatch-GFPs 1
    [setup-GFP
      set color green]
    set breed BCAM0228s 
    setup-BCAM0228
    move-to one-of patches with [pcolor = 78]
    set BCAM-bound false]
end

to degrade
  if ticks mod decay-frequency = 0 [
    if count signals > 0 [
      set todie ((count signals / 100 ) * degprob)
      ask n-of todie signals[die]
      set degraded-signal (degraded-signal + todie)]
    
    if count signal2s > 0 [
      set todie ((count signal2s / 100 ) * degprob)
      ask n-of todie signal2s[die]
      set degraded-signal (degraded-signal + todie)]
    
    if count BDSFs > 0 [
      set todie ((count BDSFs / 100 ) * degprob)
      ask n-of todie BDSFs[die]
      set degraded-signal (degraded-signal + todie)]
    
    set todie ((count GFPs / 100) * degprob)
    ask n-of todie GFPs[die]
    set degraded-GFP (degraded-GFP + todie)
    set todie 0
  ]
end


;;; procedure detecting collisions between particles in the environment ;;;
to check-for-collision 
  if count other turtles-here = 1 [                                          ;; if two turtles occupy the same current space    
    let candidate one-of other turtles-here with                             ;; choose one as a candidate
      [who < [who] of myself and myself != last-collision]                   ;; who was not the last candidate for collision
    if (candidate != nobody) and (speed > 0 or [speed] of candidate > 0)[    ;; if there is a candidate who is not stationary
      collide-with candidate                                                 ;; run the collide-with procedure with the candidate
      set last-collision candidate                                           ;; having taken part in a collision, set the asking turtle's last collision
      ask candidate [ set last-collision myself ]                            ;; and the candidate turtle's last collision
    ]
  ]
end

;;; a procedure controlling the physics of collisions between particles which are colliding ;;;
;;; the equations below take into account the masses, speeds and heading of the colliding particles and carries out momentum conservation calculations ;;;
to collide-with [ other-particle ] 
  let mass2 [mass] of other-particle
  let speed2 [speed] of other-particle
  let heading2 [heading] of other-particle
  let theta (random-float 360)
  let v1t (speed * cos (theta - heading))
  let v1l (speed * sin (theta - heading))
  let v2t (speed2 * cos (theta - heading2))
  let v2l (speed2 * sin (theta - heading2))
  let vcm (((mass * v1t) + (mass2 * v2t)) / (mass + mass2) )
  set v1t (2 * vcm - v1t)
  set v2t (2 * vcm - v2t)
  set speed sqrt ((v1t ^ 2) + (v1l ^ 2))
  set energy (0.5 * mass * speed ^ 2)
  if v1l != 0 or v1t != 0
    [ set heading (theta - (atan v1l v1t)) ]
  ask other-particle [
    set speed sqrt ((v2t ^ 2) + (v2l ^ 2))
    set energy (0.5 * mass * (speed ^ 2))
    if v2l != 0 or v2t != 0
      [ set heading (theta - (atan v2l v2t)) ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
301
18
795
453
60
50
4.0
1
10
1
1
1
0
1
1
1
-60
60
-50
50
1
1
1
ticks
30.0

BUTTON
10
10
115
43
setup-iGEM-PQS
setup-iGEM-PQS
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
118
28
181
61
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
82
182
115
initial-PQS
initial-PQS
0
1000
120
5
1
NIL
HORIZONTAL

INPUTBOX
96
485
167
545
diffuseprob
90
1
0
Number

BUTTON
10
45
115
78
setup-WT-PQS
setup-WT-PQS
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
11
485
94
545
signal-required
2
1
0
Number

PLOT
799
204
1128
386
PQS level
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"set-plot-y-range (count signal2s with [color = sky]) (count signal2s with [color = sky]) + 10" ""
PENS
"default" 1.0 0 -13791810 true "" "plot count signal2s with [color = sky]"

PLOT
798
19
1128
201
Number of Fluorescent Molecules Produced
NIL
NIL
0.0
1200.0
0.0
12.0
false
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot count GFPs"

BUTTON
10
209
117
242
setup-iGEM-BDSF
setup-iGEM-BDSF
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
245
182
278
initial-BDSF
initial-BDSF
0
1000
220
10
1
NIL
HORIZONTAL

SLIDER
10
279
182
312
initial-BCAM0228
initial-BCAM0228
0
100
27
1
1
NIL
HORIZONTAL

SLIDER
10
381
182
414
initial-ATP
initial-ATP
0
500
100
10
1
NIL
HORIZONTAL

INPUTBOX
169
485
241
545
bindprob
90
1
0
Number

SLIDER
10
150
182
183
initial-PQSA
initial-PQSA
0
50
10
1
1
NIL
HORIZONTAL

SLIDER
10
116
182
149
initial-PQSR
initial-PQSR
4
52
24
4
1
NIL
HORIZONTAL

SLIDER
10
347
182
380
initial-Pcbld
initial-Pcbld
0
100
51
1
1
NIL
HORIZONTAL

TEXTBOX
203
30
282
58
Choose a setup\nThen press GO
11
0.0
1

TEXTBOX
205
84
290
114
Number of PQS molecules
11
0.0
1

TEXTBOX
205
118
298
146
Number of PQSR receptors
11
0.0
1

TEXTBOX
204
153
299
184
Number of PQSA promoters
11
0.0
1

BUTTON
118
209
181
242
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
13
551
96
643
Number of signalling molecules required to activate the receptors
11
0.0
1

TEXTBOX
106
552
166
626
% \ndiffusion across the cell membrane 
11
0.0
1

TEXTBOX
179
552
243
627
%\nsignalling molecules bind to the receptors
11
0.0
1

TEXTBOX
205
212
288
240
Choose a setup\nThen press GO
11
0.0
1

TEXTBOX
206
248
311
277
Number of BDSF molecules
11
0.0
1

TEXTBOX
205
282
304
311
Number of BCAM0228 molecules
11
0.0
1

TEXTBOX
204
385
309
414
Number of ATP molecules
11
0.0
1

TEXTBOX
204
350
317
380
Number of Pcbld promoters
11
0.0
1

SLIDER
10
313
182
346
initial-BCAM0227
initial-BCAM0227
6
30
12
6
1
NIL
HORIZONTAL

BUTTON
558
481
737
514
NIL
make-movie-iGEM-PQS
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
558
586
737
631
NIL
movie-status
3
1
11

INPUTBOX
322
517
373
577
degprob
1
1
0
Number

TEXTBOX
382
538
488
571
% particles decay
11
0.0
1

SLIDER
322
483
490
516
decay-frequency
decay-frequency
0
500
95
1
1
ticks
HORIZONTAL

MONITOR
321
579
418
624
NIL
degraded-signal
0
1
11

MONITOR
420
579
506
624
NIL
degraded-GFP
0
1
11

TEXTBOX
346
459
520
477
Degradation Controls
13
0.0
1

TEXTBOX
557
458
783
481
Create a video of the simulation
13
0.0
1

PLOT
799
390
1129
583
Active Promoters
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -5825686 true "" "plot count PQSAs with [ready = true]"

BUTTON
558
516
737
549
NIL
make-movie-WT-PQS
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
558
551
737
584
NIL
make-movie-iGEM-BDSF
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
205
315
267
348
Number of BCAM0227\n
11
0.0
1

@#$#@#$#@
##The Lung Ranger

The Dundee 2014 iGEM project is focused on designing and testing a device that will rapidly and non-invasively identify the bacteria colonising a Cystic Fibrosis patient. Three biosensors were developed that recognise external signal molecules produced by key bacteria, all of which are known to be in sputum samples of Cystic Fibrosis patients. A quinolone signal (PQS) is produced by Pseudomonas aeruginosa, a Diffusible Signal Factor (DSF) is produced by Stenotrophomonas maltophilia, an organism that is also associated with Cystic Fibrosis lung infection in adults, and BDSF is a related, but chemically distinct molecule that is produced by Burkholderia cepacia. 

Each biosenser contains an E.coli chassis, which has been engineered to detect the signalling molecule and produce a fluorescent output.

Since our project focuses strongly on public engagement we realised that we would need to have an alternative way of explaining to the public the science behind our project. NetLogo was our answer.


##Netlogo model
The model represents a cross-section of our engineered E.coli cell.  Within the interface there are three separate models - 2 PQS models and 1 BDSF model. A model was created for DSF. However there was too much going on to follow what was happening and so is not included in this interface. 

###Model 1 - set-iGEM-PQS
This setup represents an engineered E.coli cell which detects PQS and expresses mCherry. 

####How it works
PQS (green dots) move through the cell membrane and bind PQSR (dark blue squares).  Once 2 PQS have bound the plasmid containing the promoter (magenta rings) moves towards the PQSR.  Upon reaching the PQSR, a mCherry molecule is produced (red star) and the PQS molecules released. 

The number of mCherry expressed and the number of active promoters can be seen in the graphs. 

####Things to try
The initial values for PQS, PQSR and PQSA are controlled by sliders. Try running the model with variations of the initial values. 

###Model 2 - set-iGEM-WT
This setup represents a Pseudomonas cell which detects PQS and expresses PQS (positive feedback loop)

####How it works
PQS (green dots) move through the cell membrane and bind PQSR (dark blue squares).  Once 2 PQS have bound the plasmid containing the promoter (magenta rings) moves towards the PQSR.  Upon reaching the PQSR, a PQS molecule is produced (skyblue dot) and the original PQS molecules released. 

The number of PQS expressed and the number of active promoters can be seen in the graphs. 

####Things to try
The initial values for PQS, PQSR and PQSA are controlled by sliders. Try running the model with variations of the initial values. 

###Model 3 - set-iGEM-BDSF
This setup represents an engineered E.coli cell which detects BDSF and expresses mCherry

####How it works
1.	BDSF (grey dots) bind to the transmembrane-receptor BCAM0227  (magenta rectangles spanning the cell membrane) and turn the BCAM0227 dark green.
2.	ATP (orange dots) phosphorylate BCAM0227 turning it red. ATP becomes ADP (brown dots)
3.	BCAM0228 (blue squares) can then interact with phosphorylated BCAM0227 for the phosphorelay. BCAM0228 becomes red and BCAM0227 back to magenta.  The BDSF that was bound is released.
4.	Phosphorylated BCAM0228 can interact with the promoter housed in plasmid (magenta rings) to produce a molecule of GFP (green stars)
5.	Once this has happened BCAM0228 becomes blues again
The number of GFP expressed and the number of active promoters can be seen in the graphs. 

####Things to try
The initial values for BDSF,BCAM0227,BCAM0228,PcblD and ATP are controlled by sliders. Try running the model with variations of the initial values. 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

cell
false
0
Circle -7500403 true true 3 79 142
Circle -7500403 true true 163 79 142
Rectangle -7500403 true true 72 79 237 221

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
true
0
Circle -5825686 true false 0 0 300
Circle -7500403 true true 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dna
true
7
Polygon -14835848 false true 121 75 139 75 145 78 149 81 151 83 153 84 155 87 158 90 163 95 166 100 169 103 172 107 173 111 174 112 175 116 176 120 178 126 178 133 178 138 178 143 178 146 178 149 178 153 178 158 176 163 174 167 171 171 169 175 167 178 164 184 161 188 158 193 155 195 151 198 148 202 144 204 141 206 137 209 135 210 131 210 126 210 120 210
Polygon -14835848 false true 238 76 220 76 214 79 210 82 208 84 206 85 204 88 201 91 196 96 193 101 190 104 187 108 186 112 185 113 184 117 183 121 181 127 181 134 181 139 181 144 181 147 181 150 181 154 181 159 183 164 185 168 188 172 190 176 192 179 195 185 198 189 201 194 204 196 208 199 211 203 215 205 218 207 222 210 224 211 228 211 233 211 239 211
Polygon -14835848 false true 236 76 254 76 260 79 264 82 266 84 268 85 270 88 273 91 278 96 281 101 284 104 287 108 288 112 289 113 290 117 291 121 293 127 293 134 293 139 293 144 293 147 293 150 293 154 293 159 291 164 289 168 286 172 284 176 282 179 279 185 276 189 273 194 270 196 266 199 263 203 259 205 256 207 252 210 250 211 246 211 241 211 235 211
Polygon -14835848 false true 120 75 102 75 96 78 92 81 90 83 88 84 86 87 83 90 78 95 75 100 72 103 69 107 68 111 67 112 66 116 65 120 63 126 63 133 63 138 63 143 63 146 63 149 63 153 63 158 65 163 67 167 70 171 72 175 74 178 77 184 80 188 83 193 86 195 90 198 93 202 97 204 100 206 104 209 106 210 110 210 115 210 121 210
Polygon -14835848 false true 6 76 24 76 30 79 34 82 36 84 38 85 40 88 43 91 48 96 51 101 54 104 57 108 58 112 59 113 60 117 61 121 63 127 63 134 63 139 63 144 63 147 63 150 63 154 63 159 61 164 59 168 56 172 54 176 52 179 49 185 46 189 43 194 40 196 36 199 33 203 29 205 26 207 22 210 20 211 16 211 11 211 5 211
Line -14835848 true 6 76 0 76
Line -14835848 true 7 212 -2 211
Rectangle -2674135 true false 120 75 124 150
Rectangle -2674135 true false 4 149 9 211
Rectangle -2674135 true false 150 152 155 198
Rectangle -1184463 true false 150 85 154 153
Rectangle -1184463 true false 150 85 154 153
Rectangle -1184463 true false 4 77 9 149
Rectangle -13840069 true false 82 90 86 151
Rectangle -13840069 true false 28 149 33 203
Rectangle -8630108 true false 27 82 32 151
Rectangle -8630108 true false 82 150 86 190
Rectangle -13840069 true false 236 76 240 152
Rectangle -8630108 true false 236 148 240 211
Rectangle -8630108 true false 207 150 211 199
Rectangle -13840069 true false 236 76 240 152
Rectangle -13840069 true false 207 84 211 155
Rectangle -2674135 true false 267 89 271 164
Rectangle -1184463 true false 267 138 271 200
Rectangle -1184463 true false 120 148 124 210
Rectangle -13840069 true false 400 223 405 277
Polygon -14835848 false true 295 116 294 115 296 112 300 105 315 87 315 86 307 96 299 107 294 119 293 119 292 120 292 122
Polygon -14835848 false true 294 171 293 168 295 174 302 183 315 198 315 199 307 189 299 178 292 166 293 166 292 165 292 163

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

frog top
true
0
Polygon -7500403 true true 146 18 135 30 119 42 105 90 90 150 105 195 135 225 165 225 195 195 210 150 195 90 180 41 165 30 155 18
Polygon -7500403 true true 91 176 67 148 70 121 66 119 61 133 59 111 53 111 52 131 47 115 42 120 46 146 55 187 80 237 106 269 116 268 114 214 131 222
Polygon -7500403 true true 185 62 234 84 223 51 226 48 234 61 235 38 240 38 243 60 252 46 255 49 244 95 188 92
Polygon -7500403 true true 115 62 66 84 77 51 74 48 66 61 65 38 60 38 57 60 48 46 45 49 56 95 112 92
Polygon -7500403 true true 200 186 233 148 230 121 234 119 239 133 241 111 247 111 248 131 253 115 258 120 254 146 245 187 220 237 194 269 184 268 186 214 169 222
Circle -16777216 true false 157 38 18
Circle -16777216 true false 125 38 18

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.1.0
@#$#@#$#@
setup
repeat 250 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
