;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  INITIAL DECLARATIONS  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


breed [bending-arrows bending-arrow]
breed [fixed-objects fixed-object]
breed [moving-objects moving-object]


globals [
  max-value-for-colour
  min-value-for-colour

  current-random-seed
]


patches-own [
  bending-direction
  bending-magnitude
  directions-list
  magnitudes-list
]


turtles-own [
  mass
]

moving-objects-own [
 velocity
]


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;              ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  MAIN SETUP  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;              ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to setup   ; OBSERVER
  do-housekeeping     ; This line should remain BEFORE 'random-seed this-random-seed' (see 'to do-housekeeping' for an explanation of why).

  random-seed this-random-seed     ; See 'to-report this-random-seed' for an explanation of this passage. Also, this line should remain
                                   ; AFTER 'do-housekeeping' (see 'to do-housekeeping' for an explanation of why).

  generate-fixed-objects
  generate-moving-objects

  set-values-for-colour
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  MAIN GO PROCEDURES  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to go   ; OBSERVER
  update-space
  ask moving-objects [move]
  if (collisions?) [manage-collisions]
  tick
end


to update-space   ; OBSERVER
; Patches, inteded as bits of space-time, are bended by the presence of mass. The curvature of each patch is the combined effect of the
; individual bendings that each mass imposes on the patch. These are framed as vectors with a direction and a magnitude: the former is
; expressed in NetLogo degrees, the latter is the result of 'to-report bending-propagation' (see note on that procedure for more detail).
; The patch's resulting curvature (also a vector with a direction and a magnitude) is calculated by 'to-report resulting-vector-components'.

  let origins patches with [any? turtles-here]
  let targets target-patches     ; What agents are contained in 'target-patches' depends on the value of 'light-update?' in the Interface:
                                 ; if 'light-update?' is TRUE, 'target-patches' will only contain the patches where a moving object currently
                                 ; is (i.e. only those patches that will need to compute their bending for the purpose of calculating the
                                 ; motion of moving-objects); if 'light-update' is FALSE, all patches will be in 'target-patches'.

  ask targets [
   set directions-list (list)
   set magnitudes-list (list)
   set bending-direction "none"
   set bending-magnitude 0
  ]

  ; At the moment 'origins' (i.e. some patches) are the agents executing this step, because at the moment the possibility of multiple objects
  ; (i.e. turtles) on the same patch is not ruled out. If and when there will be the implementation of objects on the same patch merging, it
  ; will become more convenient to have objects execute this passage, with the introduction of a 'reach' turtles-own variable that won't have
  ; to be computed at every step but only once upon creation (and in the occasion the mass of the object changes, e.g. through collision).
  ask origins [
    let mass-here (sum [mass] of turtles-here)
    let reach (reach-based-on-mass mass-here)
    let relevant-targets (other targets in-radius reach)

    ask relevant-targets [
     set directions-list lput (towards myself) (directions-list)
     set magnitudes-list lput (bending-propagation mass-here) (magnitudes-list)
    ]
  ]

  ; See 'to-report resulting-vector-components' for a thorough explanation of the calculations involved.
  ask targets [
    let sum-vector (resulting-vector-components directions-list magnitudes-list)
    set bending-direction (first sum-vector)
    set bending-magnitude (last sum-vector)
  ]
end


to move   ; MOVING-OBJECTS
  let dir-list (list heading bending-direction)
  let mag-list (list velocity bending-magnitude)

  let sum-vector (resulting-vector-components dir-list mag-list)

  if (first sum-vector = "none") [
   set velocity 0
   stop
  ]

  set heading (first sum-vector)
  set velocity (last sum-vector)

  forward (velocity / 50)
end



to manage-collisions   ; OBSERVER

  let bigger-one NOBODY
  let smaller-one NOBODY

  ask moving-objects [
    if (not any? other moving-objects in-radius 1) [stop]

    let other-object min-one-of (other moving-objects in-radius 1) [distance myself]

    ifelse (mass >= [mass] of other-object)
     [set bigger-one self
      set smaller-one other-object]
     [set bigger-one other-object
      set smaller-one self]

    ask bigger-one [
     let dir-list (list heading [heading] of smaller-one)
     let mag-list (list (velocity * mass) ([velocity] of smaller-one * [mass] of smaller-one))     ; Velocities are multiplied by masses because momentum is the relevant quantity here.
     let sum-vector (resulting-vector-components dir-list mag-list)

     set mass (mass + ([mass] of smaller-one * 1))     ; The reason why this mass calculation is performed at this point (i.e. in the middle of the flow
     set size size-based-on-mass                       ; of commands that manage the call of 'to-report resulting-vector-components' and the result it
                                                       ; provides) is that the old mass is needed to calculate the momentum to be given as input to the
                                                       ; 'to-report resulting-vector-components' procedure, while the new mass is needed to set the new
                                                       ; velocity value based on its results (see comment to that line).

     set heading (first sum-vector)
     set velocity (last sum-vector) / mass     ; The result is divided by mass because that part of results is not a velocity but a momentum (as the
                                               ; 'to-report resulting-vector-component' procedure has been given a list containing momenta instead of
                                               ; velocities).

     ask smaller-one [die]
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  UTILITIES AT SETUP  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to do-housekeeping   ; OBSERVER
; IMPORTANT: This procedure should NOT contain any relevant randomness.
; The reason is that, in 'to setup', the 'random-seed' command is executed after 'do-housekeeping' (otherwise
; the 'clear-all' command included in 'do-housekeeping' would clear the global variable that is set during the
; execution of 'to-report this-random-seed'; check the comment on 'to-report this-random-seed' for more detail
; on this). Given that 'do-housekeeping' has to be executed before 'random-seed' in 'to setup', it follows that
; any relevant randomness in this procedure would be out of the control of the random seed, undermining its
; usefulness.
; Note that, although asking agentsets involves some randomness (because agents in an agentset are invoked in
; random order), this can be considered not-relevant as long as always the same agents are invoked (which is,
; for example, guaranteed by invoking all agents of a type, e.g. not breeds) AND as long as what agents are
; asked to do cannot vary from iteration to iteration (for example: 'ask patches [set pcolor black]'; on the
; contrary, the command 'ask patches [set pcolor black + random 3]' introduces relevant randomness).

  clear-all
  reset-ticks

  resize-world -30 30 -30 30
  set-patch-size 9.57

  ask patches [
   set pcolor spacetime-base-colour + spacetime-shade
   set directions-list (list)
   set magnitudes-list (list)
   set bending-direction "none"
  ]

  set-default-shape fixed-objects "circle 2"
  set-default-shape moving-objects "circle"
  set-default-shape bending-arrows "arrow"
end


to generate-fixed-objects   ; OBSERVER

  if (initial-number-fixed-objects > 0) [

    create-fixed-objects 1 [
     set mass (random 20) + 10    ; 8 8

     set size size-based-on-mass
     set color fixed-objects-base-colour + objects-shade
    ]

    create-fixed-objects (initial-number-fixed-objects - 1) [
     move-to one-of patches with [not any? fixed-objects-here]

     set mass (random 8) + 8

     set size size-based-on-mass
     set color fixed-objects-base-colour + objects-shade
    ]
  ]
end


to generate-moving-objects   ; OBSERVER
  create-moving-objects initial-number-moving-objects [
   move-to min-one-of patches [bending-magnitude]

   set mass (random 4) + 2
   set velocity (random-float 4) + 0.01

   set size size-based-on-mass
   set color moving-objects-base-colour + objects-shade
  ]
end

to set-values-for-colour   ; OBSERVER
; This procedure sets the values of 'bending-magnitude' to be used as maximum and minimum
; values for the range in 'scale-color' (for when the view needs to be updated with arrows
; representing the direction and magnitude of the bending).

  ; The reason for asking these two patches to set the value of 'max-value-for-colour' is because the
  ; intention is to have 'max-value-for-colour' (which will determine the colouring of arrows based
  ; on the bending magnitude of each patch) equal to the bending magnitude that the most massive
  ; object would exert on the closest patch. The bending magnitude of such scenario can be obtained
  ; by using 'to-report bending-propagation', which is a reporting procedure built to be executed
  ; by an agent being asked by another agent (that is because it uses 'distance myself'). Hence, the
  ; need to ask two agents to set this variable; specifically, 'patch 0 0' and 'patch 0 1' have a
  ; distance of 1 between themselves (which is what mimics the bending magnitude that the most massive
  ; object would exert "on the closest patch"; while 'max [mass] of turtles' is what mimics "the most
  ; massive object").
  ; Note that this might need to change if and when it will be implemented a mechanisms (such as some
  ; form of merging of objects that are on the same patch) that will make it possible to be turtles
  ; (via 'mass'), and not patches (via 'sum [mass] of turtles-here'), invoking the 'to-report bending-propagation'
  ; procedure. In that case, in fact, the minimum distance between the involved agents will not be
  ; 1 anymore (i.e. the distance between the centers of two consecutive patches) but it will be less:
  ; something around 0.5 (i.e. the distance between a turtle standing close to the middle of one of
  ; its patch's border and the center of the closest patch).
  ask patch 0 0 [
   ask patch 0 1 [
      set max-value-for-colour (bending-propagation max [mass] of turtles)
      set min-value-for-colour (bending-propagation min [mass] of turtles) * 0.10
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                                ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  COMMUNICATION WITH INTERFACE  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                                ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to base-view   ; OBSERVER
  ask bending-arrows [
   die
  ]
end


to bending-view   ; OBSERVER
  let have-to-switch? FALSE

  if (light-update?) [
   set light-update? FALSE
   set have-to-switch? TRUE
  ]

  update-space

  if (have-to-switch?) [
   set light-update? TRUE
  ]

  ask patches with [bending-direction != "none"] [
   sprout-bending-arrows 1 [
     set heading bending-direction
     set color compute-colour
    ]
  ]

  if (fill-arrows-view?) [
    ask patches with [bending-direction = "none"] [
     sprout-bending-arrows 1 [
       set heading ([heading] of min-one-of (bending-arrows with [bending-direction != "none"]) [distance myself])
       set color black ;filling-arrows-colour + 2
      ]
    ]
  ]

  ask bending-arrows [
   set size 0.82
  ]
end


to save-this-random-seed   ; OBSERVER
  set random-seed-by-user current-random-seed
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  TO-REPORT PROCEDURES  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;                        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to-report this-random-seed
; A switch in the Interface lets the user select whether they want to provide a specific random-seed and, in case, an
; input box lets the user define it. Otherwise, a new random-seed is generated automatically with 'new-seed'.
; In both cases, the seed that is actually used is stored in the global variable 'current-random-seed', which is also
; showed in the Interface (to be precise, the value is first stored in the global variable and then, from there, it is
; reported to be used as seed).

  ifelse (select-random-seed?)
   [set current-random-seed random-seed-by-user
    report current-random-seed]
   [set current-random-seed new-seed
    report current-random-seed]
end


to-report size-based-on-mass
  report mass / 5
end


to-report target-patches
  ifelse (light-update?)
   [report patches with [any? moving-objects-here]]
   [report patches]
end


to-report reach-based-on-mass [mass-here]
  let max-ignored-bending 0.0001     ; This is the maximum level of bending magnitude to be ignored (i.e. as if
                                     ; assumed to be small enough to be irrelevant).

  report mass-here * propagation-constant / max-ignored-bending
end


to-report bending-propagation [mass-here]
; The magnitude of bending is directly proportional to mass and inversely proportion to distance, therefore
; it is written here as 'B = (m / d) * k' where 'k' is the 'propagation-constant' set in the Interface.

  report (mass-here / (distance myself)) * propagation-constant
end


to-report resulting-vector-components [dir-list mag-list]
; This reporting procedure performs the sum of vectors. It takes two lists as inputs, that together
; determine the vectors acting on a specific agent: 'dir-list' contains NetLogo degrees indicating
; the directions in which said vectors operate; 'mag-list' contains their magnitudes. The procedure
; reports one list of two items: the first being the direction of the vector resulting from the sum
; of the original vectors; the second being its magnitude.
; This procedure is used both by patches and by objects: patches use it to determine the direction
; and magnitude of their bending (with the inputs being the vectors exerted by nearby objects);
; objects use it to determine the new direction and magnitude of their movement (with the inputs
; being the vector of their own last movement and the vector of their present patch's bending, or
; alternatively the vector of their own last movement and the one of the object they collide with).

  ; Directions expressed in NetLogo angles need to be converted to trigonometric angles, as
  ; this will allow the use of the 'cos' and 'sin' functions.
  let angles-list (list)
  foreach dir-list [
   d -> set angles-list lput (heading-to-angle d) (angles-list)
  ]

  ; The directions and magnitudes of vectors are transformed into the coordinates of the vectors'
  ; head. This is done by applying 'cos' and 'sin' to the vectors' directions and multiplying them
  ; by their lengths.
  let vectors-x (list)
  let vectors-y (list)
  (foreach angles-list mag-list [
    [a m] ->
    set vectors-x lput ((cos a) * m) (vectors-x)
    set vectors-y lput ((sin a) * m) (vectors-y)
    ])


  if (not empty? vectors-x) [
   let resulting-x (sum vectors-x)
   let resulting-y (sum vectors-y)

   if (resulting-x = 0) AND (resulting-y = 0) [report (list "none" 0)]

   let bending-dir (atan resulting-x resulting-y)
   let bending-mag (length-of-resulting-vector resulting-x resulting-y)

   report (list bending-dir bending-mag)
  ]
end


to-report heading-to-angle [h]
; This, used in 'to-report resulting-vector-components', is the formula for converting NetLogo headings (i.e. 0 = N, 90 = E etc)
; to trigonometric degrees (i.e. 0 = E, 90 = N etc). This is necessary in order to apply the 'sin' and 'cos' commands (that use
; trigonometric degrees) to 'directions-list' (that stores the directions from the patch to turtles in NetLogo degrees). This is
; done by converting the latter to 'angles-list'.

  report (90 - h) mod 360
end


to-report length-of-resulting-vector [xx yy]
; This reporting procedure is used to compute the magnitude of patches' bending, with the patch being the point (0;0)
; and the other point (i.e. 'resulting-x' and 'resulting-y') being the head of the vector resulting from the sum of
; all the vectors acting on the patch.
; There are two options. Given that using 'report' immediately exits the procedure, the second option will never be
; used (even if it is not commented) unless the first option is commented.

  ; One option to perform this is to use the formula for the distance between two points in a Cartesian space, with
  ; one of the points being (0;0):
  report ((xx ^ 2) + (yy ^ 2)) ^ (1 / 2)

  ; The other option is to use 'distancexy'. Given that such command computes the distance between the agent executing
  ; the command and the coordinates given as arguments, it would require asking 'patch 0 0' to perform the calculation:
  let result 0
  ask patch 0 0 [set result distancexy xx yy]
  report result
end


to-report compute-colour
  report scale-color arrows-base-colour bending-magnitude min-value-for-colour max-value-for-colour
end
@#$#@#$#@
GRAPHICS-WINDOW
322
1
913
593
-1
-1
9.57
1
10
1
1
1
0
1
1
1
-30
30
-30
30
0
0
1
ticks
30.0

BUTTON
3
340
318
373
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
3
376
318
410
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
3
413
318
446
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
162
32
317
92
initial-number-moving-objects
7.0
1
0
Number

INPUTBOX
4
32
159
92
initial-number-fixed-objects
1.0
1
0
Number

BUTTON
918
35
1071
180
NIL
base-view
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
918
202
1071
284
NIL
bending-view
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
162
303
318
336
light-update?
light-update?
0
1
-1000

SWITCH
1073
202
1228
235
fill-arrows-view?
fill-arrows-view?
1
1
-1000

CHOOSER
1074
238
1229
283
arrows-base-colour
arrows-base-colour
0 5 9.9 15 25 35 45 55 65 75 85 95 105 115 125 135
10

CHOOSER
1232
238
1386
283
filling-arrows-colour
filling-arrows-colour
0 5 9.9 15 25 35 45 55 65 75 85 95 105 115 125 135
1

SLIDER
3
303
159
336
propagation-constant
propagation-constant
0.01
1
0.07
0.01
1
NIL
HORIZONTAL

SWITCH
5
136
157
169
select-random-seed?
select-random-seed?
0
1
-1000

INPUTBOX
163
136
318
217
random-seed-by-user
5.4913313E8
1
0
Number

MONITOR
5
172
157
217
NIL
current-random-seed
0
1
11

BUTTON
5
222
318
255
NIL
save-this-random-seed
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
11
10
161
29
Manage initial mass
15
0.0
1

TEXTBOX
9
113
148
132
Manage randomness
15
0.0
1

TEXTBOX
923
10
1073
29
Manage view
15
0.0
1

TEXTBOX
1404
10
1509
282
COLORS CODES:\nblack = 0\ngray = 5\nwhite = 9.9\nred = 15\norange = 25\nbrown = 35\nyellow = 45\ngreen = 55\nlime = 65\nturquoise = 75\ncyan = 85\nsky = 95\nblue = 105\nviolet = 115\nmagenta = 125\npink = 135
13
0.0
1

CHOOSER
1075
35
1224
80
spacetime-base-colour
spacetime-base-colour
0 5 9.9 15 25 35 45 55 65 75 85 95 105 115 125 135
0

SLIDER
1231
36
1387
69
spacetime-shade
spacetime-shade
-5
5
1.5
0.5
1
NIL
HORIZONTAL

SLIDER
1231
76
1387
109
objects-shade
objects-shade
-5
5
1.0
0.5
1
NIL
HORIZONTAL

CHOOSER
1075
85
1225
130
fixed-objects-base-colour
fixed-objects-base-colour
0 5 9.9 15 25 35 45 55 65 75 85 95 105 115 125 135
8

CHOOSER
1075
135
1226
180
moving-objects-base-colour
moving-objects-base-colour
0 5 9.9 15 25 35 45 55 65 75 85 95 105 115 125 135
6

SWITCH
6
468
115
501
collisions?
collisions?
1
1
-1000

TEXTBOX
1028
509
1178
551
error:\nfixed objects = 1\nmoving objects = 57
11
0.0
1

TEXTBOX
1185
510
1335
552
orbit:\nfixed objects = 1\nmoving objects = 60
11
0.0
1

@#$#@#$#@
## WHAT IS IT?

This model tries to reproduce gravity as per General Relativity. This means reproducing motion not according to Newtonian gravity, where masses directly experience attraction at a distance, but rather by making space-time bend as an effect of the presence of mass, with the consequence that objects moving through bent space-time will have their motion affected by its curvature.
Please note that this model only attempts an anecdotical reproduction of such principle, and it does not aim at being scientifically accurate.

## HOW IT WORKS

Patches, intended as bits of space-time, are the heart of the calculations. Each mass (turtles) will impose a bending on surrounding space-time, being such bending directed towards the mass itself, directly proportional to the quantity of mass and inversely proportional to distance from the mass. Each patch that is close enough to at least one mass will therefore experience at least one such bending. The resulting bending of the patch is the sum of such bendings, considering their different directions and magnitudes.
When some mass travels through bent space-time, its direction and velocity will be affected by the direction and magnitude of the bending of the patch it is on.

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

electro
true
8
Line -11221820 true -30 75 180 -15
Line -11221820 true 0 120 270 0
Line -11221820 true 0 180 300 45
Line -11221820 true 0 240 300 105
Line -11221820 true 0 300 300 165
Line -11221820 true 135 300 300 225
Line -11221820 true 270 300 300 285
Line -11221820 true 0 15 45 0

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
